import logging
from datetime import datetime, timedelta
from typing import Dict, Iterable, List, Union

from cache_memoize import cache_memoize

from django.contrib.auth import get_user_model
from django.core.cache import cache
from django.db import transaction
from django.utils import timezone

from lms.contrib.staff.settings import (
    STAFF_CITY_COURSE_CITY_MAP_TTL, STAFF_PROFILE_CHUNK_SIZE, STAFF_PROFILE_INACTIVE_LIMIT, STAFF_PROFILE_INACTIVE_TIME,
    STAFF_PROFILE_REFRESH_TIME, STAFF_PROFILE_UPDATED_CACHE_KEY,
)

from ..utils import chunks
from .loaders import (
    BaseProfileLoader, Counts, StaffDismissedProfileLoader, StaffGroupLoader, StaffGroupProfileLoader,
    StaffHrPartnersProfileLoader, StaffLeadershipJoinedProfileLoader, StaffLeadershipProfileLoader, StaffOfficeLoader,
    StaffProfileProfileLoader, StaffUserLoader,
)
from .models import StaffCity, StaffProfile

log = logging.getLogger(__name__)
User = get_user_model()


def profile_is_updated(user_pk: int):
    return cache.get(STAFF_PROFILE_UPDATED_CACHE_KEY.format(user_pk), False)


def set_profile_updated(user_pk: Union[Iterable[int], int], timeout=None):
    if isinstance(user_pk, int):
        user_pk = [user_pk]
    if timeout is None:
        timeout = STAFF_PROFILE_REFRESH_TIME
    for pk in user_pk:
        cache.set(STAFF_PROFILE_UPDATED_CACHE_KEY.format(pk), True, timeout=int(timeout))


def update_profiles(uids: Iterable[int], loaders_dict: Dict[str, BaseProfileLoader]):
    counters = {loader: Counts() for loader in loaders_dict}
    chunk_size = min(500, STAFF_PROFILE_CHUNK_SIZE)
    for chunk in chunks(uids, chunk_size):
        for name, loader in loaders_dict.items():
            current_loader = loader(uids=chunk)
            current_loader.update()
            counters[name] += current_loader.counts
            log.info("%s: %r", name, current_loader.counts)
        set_profile_updated(chunk)
    counters_summary = "; ".join(f"{loader}: {counters[loader]}" for loader in counters)
    log.info("Full staff profiles sync counters summary: %s", counters_summary)


def update_inactive_profiles() -> None:
    """
    Обновляет профили неактивных пользователей
    """
    inactive_time = timezone.now() - timedelta(seconds=STAFF_PROFILE_INACTIVE_TIME)
    queryset = StaffProfile.objects.filter(modified__lt=inactive_time, user__yauid__isnull=False)

    if not queryset.exists():
        log.info('No inactive profiles found')
        return

    for chunk_uids in chunks(
        queryset.values_list('user__yauid', flat=True)[:STAFF_PROFILE_INACTIVE_LIMIT], STAFF_PROFILE_CHUNK_SIZE,
    ):
        StaffDismissedProfileLoader(chunk_uids).update()


def load_staff_profile(user_pk: int, force_load: bool = False) -> bool:
    """
    Загружает профиль пользователя со стаффа

    При загрузке проверяет актуальность профиля по полю modified,
    если не выставлен параметр force_load=True

    :param user_pk: ID пользователя
    :param force_load: принудительная загрузка
    :return: True при успешной загрузке, False - при отмене загрузки
    """
    profile = StaffProfile.objects.filter(user_id=user_pk, user__yauid__isnull=False).first()

    if not profile:
        log.info('No profile found: %d', user_pk)
        return False

    if not force_load:
        expired_time = timezone.now() - timedelta(seconds=STAFF_PROFILE_REFRESH_TIME)

        if profile.modified >= expired_time:
            timeout = (profile.modified - expired_time).total_seconds()
            set_profile_updated(user_pk, timeout=timeout)
            log.info('Profile %d already updated', user_pk)
            return False

    loaders = [
        StaffProfileProfileLoader,
        StaffGroupProfileLoader,
        StaffHrPartnersProfileLoader,
        StaffLeadershipProfileLoader,
        StaffLeadershipJoinedProfileLoader,
    ]

    with transaction.atomic():
        for loader in loaders:
            loader(profile.uid).update()

    set_profile_updated(user_pk)

    return True


def get_native_languages():
    return (
        StaffProfile.objects
        .exclude(language_native='')
        .values_list('language_native', flat=True)
        .distinct()
    )


@cache_memoize(timeout=STAFF_CITY_COURSE_CITY_MAP_TTL)
def get_course_city_staff_city_map():
    """
    Возвращает маппинг городов курсов на города со Стаффа

    :return:
    """
    from lms.courses.models import CourseCity

    staff_cities_name_id_map = {
        city_name.strip().lower(): city_id
        for city_id, city_name in StaffCity.objects.values_list('id', 'name_ru')
    }
    course_cities_name_id_map = {
        city_name.strip().lower(): city_id
        for city_id, city_name in CourseCity.objects.values_list('id', 'name')
    }

    course_city_for_staff_city_map = {
        staff_city_id: course_cities_name_id_map.get(staff_city_name)
        for staff_city_name, staff_city_id in staff_cities_name_id_map.items()
    }

    return course_city_for_staff_city_map


def load_staff_offices(**kwargs):
    """
    Загружает все офисы со Стаффа

    :param kwargs:
    :return:
    """
    StaffOfficeLoader(**kwargs).update()


def load_staff_groups(**kwargs):
    """
    Загружает все департаменты со Стаффа

    :param kwargs:
    :return:
    """
    StaffGroupLoader(**kwargs).update()


def load_staff_users(logins: Iterable[str]) -> List[User]:
    """
    Находит/создает пользователей по логинам, получая информацию со стаффа
    Возвращает список найденных/созданных пользователей
    """
    return StaffUserLoader(logins=logins).get_or_create()


def load_staff_profile_groups(uids: Union[Iterable[int], int], **kwargs):
    """
    Загружает список департаментов для переданных пользователей со Стаффа
    """
    return StaffGroupProfileLoader(uids, **kwargs).update()


def load_staff_profiles(**kwargs):
    log.info('Start staff profiles sync')

    kwargs.setdefault('limit', 5000)

    timestamps = {
        'start': datetime.now()
    }

    current_uids = set(
        User.objects.filter(
            staffprofile__is_dismissed=False,
            **{f"{User.YAUID_FIELD}__isnull": False}
        ).values_list(User.YAUID_FIELD, flat=True)
    )
    log.info('Found UIDs in DB: %i', len(current_uids))

    log.info('Updating staff profiles...')
    timestamps['profile_updating'] = datetime.now()
    profile_loader = StaffProfileProfileLoader(**kwargs)
    profile_loader.update_or_create()

    log.info("profiles: %r", profile_loader.counts)

    log.info('Updating additional staff data...')
    timestamps['additional_data_loading'] = datetime.now()
    loaders = {
        'Groups': StaffGroupProfileLoader,
        'Partners': StaffHrPartnersProfileLoader,
        'Leaderships': StaffLeadershipProfileLoader,
        'Leadership joined': StaffLeadershipJoinedProfileLoader,
    }

    update_profiles(profile_loader.loaded_uids, loaders)

    timestamps['dismissed_profiles_updating'] = datetime.now()
    not_updated_uids = current_uids.difference(profile_loader.loaded_uids)
    if not_updated_uids:
        log.info('Updating potential dismissed profiles...')
        update_profiles(not_updated_uids, {'Dismissed': StaffDismissedProfileLoader})

    timestamps['end'] = datetime.now()

    summary = f"""Full staff profiles sync summary:
    Total time: {timestamps['end'] - timestamps['start']}
    Found UIDs in DB: {len(current_uids)}
    Loaded UIDs from Staff: {len(profile_loader.loaded_uids)}
    Processed profiles: {profile_loader.counts['processed']}
    Created users: {profile_loader.counts['user_created']}
    Profiles processing time: {timestamps['additional_data_loading'] - timestamps['profile_updating']}
    Additional data processing time: {timestamps['dismissed_profiles_updating'] - timestamps['additional_data_loading']}
    Potential dismissed profiles updating time: {timestamps['end'] - timestamps['dismissed_profiles_updating']}
    """
    log.info(summary)
