from django_filters import rest_framework as filters
from drf_spectacular.utils import extend_schema

from django.utils.translation import gettext

from rest_framework.response import Response

from lms.core.views.pagination import LimitOffsetAllPagination
from lms.core.views.viewsets import LabModelViewSet

from ..models import StaffCity, StaffCountry, StaffGroup, StaffOffice
from ..serializers import (
    StaffCityListLabSerializer, StaffCountryListLabSerializer, StaffGroupListLabSerializer,
    StaffLanguageListLabSerializer, StaffOfficeListLabSerializer,
)
from ..services import get_native_languages


class StaffLanguageLabViewSet(LabModelViewSet):
    serializer_class = StaffLanguageListLabSerializer
    queryset = get_native_languages()
    pagination_class = None

    @extend_schema(
        summary=gettext("Список языков со Стаффа")
    )
    def list(self, request, *args, **kwargs):
        queryset = self.get_queryset()

        data = [{'name': x} for x in sorted(queryset)]
        serializer = self.get_serializer(data, many=True)
        return Response(serializer.data)


class StaffCountryLabViewSet(LabModelViewSet):
    serializer_class = StaffCountryListLabSerializer
    queryset = StaffCountry.objects.all()
    pagination_class = None

    @extend_schema(
        summary=gettext("Список стран со Стаффа")
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)


class StaffCityLabViewSet(LabModelViewSet):
    serializer_class = StaffCityListLabSerializer
    queryset = StaffCity.objects.all()
    pagination_class = None

    @extend_schema(
        summary=gettext("Список городов со Стаффа")
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)


class StaffOfficeLabViewSet(LabModelViewSet):
    serializer_class = StaffOfficeListLabSerializer
    queryset = StaffOffice.objects.all()
    pagination_class = None

    @extend_schema(
        summary=gettext("Список офисов со Стаффа")
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)


class StaffGroupFilter(filters.FilterSet):
    # используем полнотекст из postgres
    # https://docs.djangoproject.com/en/2.2/ref/contrib/postgres/search/#the-search-lookup
    name = filters.CharFilter(field_name='name', lookup_expr='search')


class StaffGroupLabViewSet(LabModelViewSet):
    serializer_class = StaffGroupListLabSerializer
    queryset = StaffGroup.objects.all()
    pagination_class = LimitOffsetAllPagination
    filter_backends = (filters.DjangoFilterBackend,)
    filterset_class = StaffGroupFilter

    @extend_schema(
        summary=gettext("Список подразделений")
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)

    @extend_schema(
        summary=gettext("Информация о подразделении")
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)
