from django.utils.translation import gettext_lazy as _

from rest_framework import serializers

from .models import Tag
from .services import update_user_tags
from .utils import normalize_tag_name


# API
# ===
class TagListSerializer(serializers.ModelSerializer):
    """
    Информация о теге в списке
    """
    class Meta:
        model = Tag
        fields = (
            'id', 'name', 'normalized_name',
        )
        read_only_fields = fields


class UserTagUpdateSerializer(serializers.Serializer):
    """
    Обновлённые интересующие пользователя теги
    """
    tags = serializers.ListSerializer(child=serializers.CharField())

    def validate_tags(self, value):
        if value and len(value) != len({normalize_tag_name(tag) for tag in value}):
            raise serializers.ValidationError(_("Присутствуют дублирующиеся теги"))

        return value

    def create(self, validated_data):
        tags = validated_data.get('tags')
        user = self.context['request'].user
        update_user_tags(user=user, tags=tags)
        return tags


class UserTagListSerializer(serializers.ModelSerializer):
    """
    Информация о теге в списке интересов пользователя
    """
    class Meta:
        model = Tag
        fields = (
            'id', 'name', 'normalized_name',
        )
        read_only_fields = fields


# LABAPI
# ===
class TagDetailLabSerializer(serializers.ModelSerializer):
    """
    Информация о теге
    """
    class Meta:
        model = Tag
        fields = (
            'id', 'name', 'normalized_name', 'created', 'modified',
        )
        read_only_fields = fields


class TagListLabSerializer(serializers.ModelSerializer):
    """
    Информация о теге в списке
    """
    class Meta:
        model = Tag
        fields = (
            'id', 'name', 'normalized_name', 'created', 'modified',
        )
        read_only_fields = fields


class TagCreateLabSerializer(serializers.ModelSerializer):
    """
    Информация о теге при создании
    """
    class Meta:
        model = Tag
        fields = (
            'name',
        )


class TagUpdateLabSerializer(serializers.ModelSerializer):
    """
    Информация о теге при обновлении
    """
    class Meta:
        model = Tag
        fields = (
            'name',
        )
