from django_filters import rest_framework as filters
from drf_spectacular.utils import extend_schema

from django.contrib.auth import get_user_model
from django.utils.translation import gettext

from rest_framework import status
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response

from lms.core.views.pagination import LimitOffsetAllPagination
from lms.core.views.viewsets import APIModelViewSet

from ..filtersets import TagFilterSet
from ..models import Tag
from ..serializers import TagListSerializer, UserTagListSerializer, UserTagUpdateSerializer

User = get_user_model()


class TagViewSet(APIModelViewSet):
    serializer_class = TagListSerializer
    queryset = Tag.objects.all()
    filter_backends = [filters.DjangoFilterBackend]
    filterset_class = TagFilterSet
    pagination_class = LimitOffsetAllPagination

    @extend_schema(
        responses={200: TagListSerializer},
        summary=gettext("Список тегов")
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)


class UserTagViewSet(APIModelViewSet):
    serializer_class = UserTagListSerializer
    serializer_classes = {
        'update': UserTagUpdateSerializer,
    }
    queryset = Tag.objects.all()
    permission_classes = APIModelViewSet.permission_classes + [
        IsAuthenticated,
    ]
    pagination_class = LimitOffsetAllPagination

    def get_queryset(self):
        queryset = super().get_queryset()
        user = getattr(self.request, 'user', None)
        if user and isinstance(user, User):
            return queryset.filter(
                user_tags__user_id=user.id,
            )
        return queryset.none()

    @extend_schema(
        responses={200: UserTagListSerializer},
        summary=gettext("Список интересов пользователя")
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)

    @extend_schema(
        responses={204: None},
        summary=gettext("Обновить список интересов пользователя")
    )
    def update(self, request, *args, **kwargs):
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        self.perform_update(serializer)
        return Response(status=status.HTTP_204_NO_CONTENT)
