from tqdm import tqdm

from django.conf import settings
from django.core.management.base import BaseCommand
from django.db import transaction

from ...models import (
    EnrolledUserTrackerIssue, EnrollmentQueue, EnrollmentTracker, EnrollmentTrackerIssue, EnrollmentTrackerQueue,
    QueueType, TicketTemplate, TrackerHook, TrackerHookEvent,
)

DEFAULT_QUEUE_INITIAL_STATE = ''


class DryRunException(Exception):
    pass


class Command(BaseCommand):
    help = "Migrate enrollment issues"

    def migrate_enrollment_issues(self, **options):
        try:
            with transaction.atomic():
                default_ticket_template_id = options['ticket_template_id']
                progress = options.get('progress', False)
                default_ticket_template = TicketTemplate.objects.filter(id=default_ticket_template_id).first()
                if default_ticket_template is None:
                    raise Exception('TicketTemplate %d not found' % default_ticket_template_id)

                # мигрируем очереди
                enrollment_queues_to_create_map = {}
                existing_enrollment_queues_map = {
                    (
                        enrollment_queue.name,
                        enrollment_queue.issue_type,
                        enrollment_queue.initial_state,
                        enrollment_queue.template_id,
                    ): enrollment_queue
                    for enrollment_queue
                    in EnrollmentQueue.objects.all()
                }
                self.stdout.write('Look through queues')

                enrollment_tracker_queues = EnrollmentTrackerQueue.objects.select_related('template', 'enrollment')
                enrollment_tracker_queues = (
                    tqdm(enrollment_tracker_queues, total=EnrollmentTrackerQueue.objects.count()) if progress
                    else enrollment_tracker_queues
                )
                for enrollment_tracker_queue in enrollment_tracker_queues:
                    if enrollment_tracker_queue.template is None:
                        template = default_ticket_template
                    else:
                        template = enrollment_tracker_queue.template

                    enrollment_queue_key = (
                        enrollment_tracker_queue.name,
                        enrollment_tracker_queue.issue_type,
                        DEFAULT_QUEUE_INITIAL_STATE,
                        template.id,
                    )

                    if (
                        enrollment_queue_key not in existing_enrollment_queues_map and
                        enrollment_queue_key not in enrollment_queues_to_create_map
                    ):
                        enrollment_queue = EnrollmentQueue(
                            name=enrollment_tracker_queue.name,
                            issue_type=enrollment_tracker_queue.issue_type,
                            initial_state=DEFAULT_QUEUE_INITIAL_STATE,
                            template_id=template.id,
                        )
                        enrollment_queues_to_create_map[enrollment_queue_key] = enrollment_queue

                # Can't bulk create a multi-table inherited model, поэтому так
                self.stdout.write("Create queues")
                enrollment_queues = enrollment_queues_to_create_map.values()
                enrollment_queues = tqdm(enrollment_queues) if progress else enrollment_queues
                for enrollment_queue in enrollment_queues:
                    enrollment_queue.save()

                # мигрируем связи очередей с методами зачисления
                enrollment_tracker_to_create_map = {}
                existing_enrollment_queues_map = {
                    (
                        enrollment_queue.name,
                        enrollment_queue.issue_type,
                        enrollment_queue.initial_state,
                        enrollment_queue.template_id,
                    ): enrollment_queue
                    for enrollment_queue
                    in EnrollmentQueue.objects.all()
                }
                existing_enrollment_tracker_map = {
                    enrollment_tracker.enrollment_tracker_queue_id: enrollment_tracker
                    for enrollment_tracker in EnrollmentTracker.objects.all()
                    if enrollment_tracker.enrollment_tracker_queue_id is not None
                }
                self.stdout.write("Look through enrollment queues")
                enrollment_tracker_queues = EnrollmentTrackerQueue.objects.select_related('template', 'enrollment')
                enrollment_tracker_queues = (
                    tqdm(enrollment_tracker_queues, total=EnrollmentTrackerQueue.objects.count()) if progress
                    else enrollment_tracker_queues
                )
                for enrollment_tracker_queue in enrollment_tracker_queues:
                    enrollment_id = enrollment_tracker_queue.enrollment_id
                    enrollment_queue_key = (
                        enrollment_tracker_queue.name,
                        enrollment_tracker_queue.issue_type,
                        DEFAULT_QUEUE_INITIAL_STATE,
                        (
                            default_ticket_template.id if enrollment_tracker_queue.template is None
                            else enrollment_tracker_queue.template_id
                        ),
                    )
                    enrollment_queue = existing_enrollment_queues_map[enrollment_queue_key]
                    enrollment_tracker_key = (enrollment_id, enrollment_queue.id)
                    if (
                        enrollment_tracker_key not in enrollment_tracker_to_create_map and
                        enrollment_tracker_queue.id not in existing_enrollment_tracker_map
                    ):
                        enrollment_tracker_to_create_map[enrollment_tracker_key] = (
                            EnrollmentTracker(
                                enrollment_id=enrollment_id,
                                queue_id=enrollment_queue.id,
                                is_default=enrollment_tracker_queue.is_default,
                                enrollment_tracker_queue=enrollment_tracker_queue,
                            )
                        )

                EnrollmentTracker.objects.bulk_create(
                    objs=enrollment_tracker_to_create_map.values(),
                    batch_size=settings.BULK_BATCH_SIZE_DEFAULT,
                )

                # мигрируем тикеты
                enrollment_tracker_queue_to_enrollment_tracker_map = {
                    enrollment_tracker.enrollment_tracker_queue_id: enrollment_tracker.queue
                    for enrollment_tracker in EnrollmentTracker.objects.select_related('queue')
                }
                enrolled_user_tracker_issues_to_create_map = {}
                existing_enrolled_user_tracker_issue_map = {
                    enrolled_user_tracker_issue.enrollment_tracker_issue_id: enrolled_user_tracker_issue
                    for enrolled_user_tracker_issue in EnrolledUserTrackerIssue.objects.all()
                    if enrolled_user_tracker_issue.enrollment_tracker_issue_id is not None
                }
                self.stdout.write("Look through issues")
                enrollment_tracker_issues = EnrollmentTrackerIssue.objects.select_related('enrolled_user')
                enrollment_tracker_issues = (
                    tqdm(enrollment_tracker_issues, total=EnrollmentTrackerIssue.objects.count()) if progress
                    else enrollment_tracker_issues
                )
                for enrollment_tracker_issue in enrollment_tracker_issues:
                    queue = enrollment_tracker_queue_to_enrollment_tracker_map[enrollment_tracker_issue.queue_id]
                    issue_key = f'{queue.name}-{enrollment_tracker_issue.issue_number}'
                    enrolled_user_tracker_issue_key = (
                        enrollment_tracker_issue.enrolled_user_id,
                        queue.id,
                        issue_key,
                        enrollment_tracker_issue.status
                    )
                    if (
                        enrollment_tracker_issue.id not in existing_enrolled_user_tracker_issue_map and
                        enrolled_user_tracker_issue_key not in enrolled_user_tracker_issues_to_create_map
                    ):
                        enrolled_user_tracker_issue = EnrolledUserTrackerIssue(
                            enrolled_user=enrollment_tracker_issue.enrolled_user,
                            queue_type=QueueType.ENROLLMENT,
                            queue=queue,
                            is_default=enrollment_tracker_issue.queue.is_default,
                            issue_key=issue_key,
                            issue_status=enrollment_tracker_issue.status,
                            status=EnrolledUserTrackerIssue.Status.SUCCESS,
                            enrollment_tracker_issue=enrollment_tracker_issue,
                        )
                        enrolled_user_tracker_issues_to_create_map[enrolled_user_tracker_issue_key] = (
                            enrolled_user_tracker_issue
                        )

                # Can't bulk create a multi-table inherited model, поэтому так
                self.stdout.write('create issues')
                enrolled_user_tracker_issues_to_create = enrolled_user_tracker_issues_to_create_map.values()
                enrolled_user_tracker_issues_to_create = (
                    tqdm(enrolled_user_tracker_issues_to_create) if progress
                    else enrolled_user_tracker_issues_to_create
                )
                for enrolled_user_tracker_issue_to_create in enrolled_user_tracker_issues_to_create:
                    enrolled_user_tracker_issue_to_create.save()

                # мигрируем хуки
                existing_tracker_hook_map = {
                    tracker_hook.tracker_hook_event_id: tracker_hook
                    for tracker_hook in TrackerHook.objects.all()
                    if tracker_hook.tracker_hook_event_id is not None
                }
                existing_enrolled_user_tracker_issue_map = {
                    enrolled_user_tracker_issue.enrollment_tracker_issue_id: enrolled_user_tracker_issue
                    for enrolled_user_tracker_issue in EnrolledUserTrackerIssue.objects.all()
                    if enrolled_user_tracker_issue.enrollment_tracker_issue_id is not None
                }
                tracker_hooks_to_create_map = {}
                self.stdout.write('look through tracker hooks events')
                tracker_hook_events = TrackerHookEvent.objects.select_related('issue')
                tracker_hook_events = (
                    tqdm(tracker_hook_events, total=TrackerHookEvent.objects.count()) if progress
                    else tracker_hook_events
                )
                for tracker_hook_event in tracker_hook_events:
                    if (
                        tracker_hook_event.id not in existing_tracker_hook_map and
                        tracker_hook_event.id not in tracker_hooks_to_create_map
                    ):
                        tracker_hook = TrackerHook(
                            issue=(
                                None if tracker_hook_event.issue_id is None
                                else existing_enrolled_user_tracker_issue_map[tracker_hook_event.issue_id]
                            ),
                            data=tracker_hook_event.request_body,
                            status=tracker_hook_event.status,
                            result=tracker_hook_event.process_status_result,
                            tracker_hook_event=tracker_hook_event,
                        )
                        tracker_hooks_to_create_map[tracker_hook_event.id] = tracker_hook

                TrackerHook.objects.bulk_create(
                    objs=tracker_hooks_to_create_map.values(),
                    batch_size=settings.BULK_BATCH_SIZE_DEFAULT,
                )

                self.stdout.write(f"""
                enrollment_queues_to_create_map={enrollment_queues_to_create_map}
                enrollment_tracker_to_create_map={enrollment_tracker_to_create_map}
                enrolled_user_tracker_issues_to_create_map={enrolled_user_tracker_issues_to_create_map}
                tracker_hooks_to_create_map={tracker_hooks_to_create_map}
                """)

                # миграция очередей
                if options.get('dry_run', False):
                    raise DryRunException

        except DryRunException:
            self.stdout.write('Rollback')
        else:
            self.stdout.write('Commit')

    def add_arguments(self, parser):
        parser.add_argument(
            '--dry-run', action='store_true', dest='dry_run',
            help='Run without commit to db',
        )
        parser.add_argument(
            '--ticket-template-id', '-t', dest='ticket_template_id', type=int, required=True,
            help='Default ticket template id',
        )
        parser.add_argument(
            '--progress', action='store_true', dest='progress',
            help='Output generator progress',
        )

    def handle(self, *args, **options):
        self.migrate_enrollment_issues(**options)
        self.stdout.write("Done\n")
