import warnings

from model_utils.models import TimeStampedModel, UUIDModel
from ordered_model.models import OrderedModel, OrderedModelQuerySet
from simple_history.models import HistoricalRecords

from django.contrib.auth import get_user_model
from django.contrib.postgres.fields import JSONField
from django.core.exceptions import ValidationError
from django.db import models
from django.utils.translation import gettext_lazy as _

from lms.classrooms.models import Classroom, StudentSlot
from lms.core.models.mixins import ActiveFilterMixin
from lms.courses.models import CourseCategory
from lms.enrollments.models import EnrolledUser, Enrollment

User = get_user_model()


class TicketTemplate(TimeStampedModel):
    name = models.CharField(_("название"), max_length=255)
    summary = models.TextField(_("название тикета"))
    description = models.TextField(_("описание тикета"), blank=True)

    is_active = models.BooleanField(_("активен"), default=True)

    history = HistoricalRecords()

    class Meta:
        ordering = ('-created',)
        verbose_name = _("шаблон тикета")
        verbose_name_plural = _("шаблоны тикетов")
        permissions = (
            ('import_tickettemplate', _('Can import ticket templates')),
            ('export_tickettemplate', _('Can export ticket templates')),
        )

    def __str__(self):
        return self.name


class TicketFieldTemplateQuerySet(ActiveFilterMixin, models.QuerySet):
    pass


class TicketFieldTemplate(TimeStampedModel):
    template = models.ForeignKey(
        TicketTemplate,
        verbose_name=_("шаблон тикета"),
        related_name='fields',
        on_delete=models.CASCADE,
    )
    field_name = models.CharField(_("название поля"), max_length=500)
    value = models.TextField(_("значение"), blank=True)
    allow_null = models.BooleanField(_("null?"), default=False)
    allow_json = models.BooleanField(_("json?"), default=False)

    is_active = models.BooleanField(_("активно"), default=True)

    objects = TicketFieldTemplateQuerySet.as_manager()

    class Meta:
        ordering = ('field_name',)
        unique_together = [
            ['template', 'field_name'],
        ]
        verbose_name = _("шаблон для поля тикета")
        verbose_name_plural = _("шаблоны для полей тикетов")
        permissions = (
            ('import_ticketfieldtemplate', _('Can import ticket field templates')),
            ('export_ticketfieldtemplate', _('Can export ticket field templates')),
        )

    def __str__(self):
        return f"[{self.template_id}] {self.field_name}"


class EnrollmentTrackerQueueQuerySet(ActiveFilterMixin, OrderedModelQuerySet):
    pass


# DEPRECATED
class EnrollmentTrackerQueue(OrderedModel, TimeStampedModel):
    name = models.CharField(
        verbose_name=_("имя очереди"),
        max_length=255,
    )
    enrollment = models.ForeignKey(
        Enrollment,
        verbose_name=_("зачисление"),
        related_name='tracker_queues',
        on_delete=models.PROTECT,
    )
    is_default = models.BooleanField(
        verbose_name=_("очередь по умолчанию"),
        default=False,
    )
    summary = models.CharField(
        verbose_name=_("заголовок тикета"),
        max_length=255,
        default="{course} - {first_name} {last_name} ({login})",
        help_text='Доступны параметры: course - название курса, first_name - имя, last_name - фамилия, login - логин',
    )
    description = models.TextField(
        verbose_name=_("описание тикета"),
        blank=True,
    )
    issue_type = models.CharField(
        verbose_name=_("тип тикета"),
        max_length=255,
    )
    accepted_status = models.CharField(
        verbose_name=_("статус для подтверждения заявки"),
        max_length=255,
        default='accepted',
        null=True,
        blank=True,
    )
    rejected_status = models.CharField(
        verbose_name=_("статус для отклонения заявки"),
        max_length=255,
        default='rejected',
        null=True,
        blank=True,
    )
    cancelled_status = models.CharField(
        verbose_name=_("статус для отмены заявки"),
        max_length=255,
        default='cancelled',
        null=True,
        blank=True,
    )
    tracker_pulling_enabled = models.BooleanField(
        verbose_name=_("запрашивать статусы из трекера"),
        help_text=_("Периодически опрашиваем трекер на наличие изменений статуса по тикетам (long-polling)"),
        default=True,
    )
    template = models.ForeignKey(
        TicketTemplate,
        verbose_name=_("шаблон тикета"),
        related_name='enrollment_tracker_queues',
        null=True,
        blank=True,
        on_delete=models.SET_NULL,
    )
    is_active = models.BooleanField(_("активна"), default=True)

    objects = EnrollmentTrackerQueueQuerySet.as_manager()

    history = HistoricalRecords()

    def __str__(self):
        return self.name

    def clean(self):
        if self.enrollment.enroll_type != self.enrollment.TYPE_TRACKER:
            raise ValidationError({
                'enrollment': ValidationError(
                    _("Данный механизм зачисления не может содержать очередей в трекере"),
                    code='non_tracker_enrollment'
                ),
            })
        if self.is_default and self.tracker_pulling_enabled:
            status_errors = {}
            for field in ['accepted_status', 'rejected_status', 'cancelled_status']:
                if not getattr(self, field):
                    status_errors[field] = ValidationError(
                        _("Для очереди сотрудника поле должно быть заполнено"),
                        code='required',
                    )

            if status_errors:
                raise ValidationError(status_errors)

    @property
    def has_issues(self):
        return self.enrollment_tracker_issues.exists()

    @property
    def is_last_queue(self):
        qs = EnrollmentTrackerQueue.objects\
            .filter(enrollment_id=self.enrollment_id)\
            .exclude(pk=self.pk)

        return not qs.exists()

    @property
    def can_delete(self) -> bool:
        return not self.has_issues

    def save(self, *args, **kwargs):
        self.clean()
        return super().save(*args, **kwargs)

    def delete(self, *args, **kwargs):
        if self.is_default and not self.is_last_queue:
            raise ValidationError(
                _("Чтобы удалить данную очередь, нужно сначала удалить все остальные очереди"),
                code='default_queue_is_not_last',
            )
        if not self.can_delete:
            raise ValidationError(
                _("Нельзя удалить очередь, для который уже были созданы заявки на зачисление"),
                code='queue_has_issues',
            )
        return super().delete(*args, **kwargs)

    order_with_respect_to = 'enrollment'

    class Meta:
        verbose_name = _("очередь для зачисления")
        verbose_name_plural = _("очереди для зачисления")
        ordering = ("order",)
        permissions = (
            ('import_enrollmenttrackerqueue', _('Can import enrollment tracker queues')),
            ('export_enrollmenttrackerqueue', _('Can export enrollment tracker queues')),
        )


class EnrollmentTrackerIssueQuerySet(ActiveFilterMixin, models.QuerySet):
    pass


# DEPRECATED
class EnrollmentTrackerIssue(TimeStampedModel):
    enrolled_user = models.ForeignKey(
        EnrolledUser,
        verbose_name=_("заявка на зачисление"),
        related_name='enrollment_tracker_issues',
        on_delete=models.PROTECT,
    )
    queue = models.ForeignKey(
        EnrollmentTrackerQueue,
        verbose_name=_("очередь"),
        related_name='enrollment_tracker_issues',
        on_delete=models.PROTECT,
    )
    issue_number = models.IntegerField(
        verbose_name=_("номер тикета"),
        null=True,
        blank=True,
    )
    got_status_from_startrek = models.DateTimeField(
        verbose_name=_("дата получения статуса из стартрека"),
        null=True,
        blank=True,
    )
    status = models.CharField(
        verbose_name=_("статус"),
        max_length=255,
        blank=True,
    )
    status_processed = models.BooleanField(
        verbose_name=_("статус обработан"),
        default=False,
    )
    is_active = models.BooleanField(_("активен"), default=True)

    objects = EnrollmentTrackerIssueQuerySet.as_manager()

    def _issue(self) -> str:
        return f'{self.queue.name}-{self.issue_number}'

    _issue.short_description = _("Тикет")
    issue = property(_issue)

    @property
    def enrollment(self):
        return self.enrolled_user.enrollment

    @property
    def is_accepted(self):
        return self.status == self.queue.accepted_status

    @property
    def is_rejected(self):
        return self.status == self.queue.rejected_status

    @property
    def is_cancelled(self):
        return self.status == self.queue.cancelled_status

    class Meta:
        unique_together = ('enrolled_user', 'queue')
        verbose_name = _("тикет на зачисление")
        verbose_name_plural = _("тикеты на зачисление")
        permissions = (
            ('import_enrollmenttrackerissue', _('Can import enrollment tracker issue')),
            ('export_enrollmenttrackerissue', _('Can export enrollment tracker issue')),
        )

    def __str__(self):
        return f'[{self.pk}] {self.issue}'

    def clean_enrollment(self):
        if self.enrolled_user.enrollment_id != self.queue.enrollment_id:
            message = "Механизм зачисления в заявке должен совпадать с механизмом зачисления в очереди"
            raise ValidationError(
                {
                    'enrolled_user': (message),
                    'queue': (message),
                },
            )

    def clean(self):
        self.clean_enrollment()


class TrackerCourseCategory(TimeStampedModel):
    category = models.OneToOneField(
        to=CourseCategory,
        verbose_name=_('Категория'),
        related_name='tracker_course_category',
        on_delete=models.CASCADE,
    )
    tracker_course_category_id = models.IntegerField(_('ID категории'))

    def __str__(self):
        return self.category.name

    class Meta:
        verbose_name = _('ID категории в трекере')
        verbose_name_plural = _('ID категорий в трекере')


# DEPRECATED
class TrackerHookEvent(TimeStampedModel):
    issue = models.ForeignKey(
        EnrollmentTrackerIssue, verbose_name=_('тикет'), null=True, blank=True, on_delete=models.CASCADE,
    )
    request_body = JSONField(verbose_name=_('тело запроса'), null=True, blank=True)

    STATUS_PENDING = 'pending'
    STATUS_SUCCESS = 'success'
    STATUS_ERROR = 'error'

    STATUS_CHOICES = (
        (STATUS_PENDING, _("в обработке")),
        (STATUS_SUCCESS, _("обработан")),
        (STATUS_ERROR, _("ошибка")),
    )

    status = models.CharField(
        verbose_name=_('статус обработки хука'), max_length=20, choices=STATUS_CHOICES, default=STATUS_PENDING,
    )
    process_status_result = models.CharField(verbose_name=_('результат обработки статуса'), max_length=255, blank=True)

    def __str__(self):
        return f'{self.issue} ({self.get_status_display()})'

    class Meta:
        verbose_name = _("Хук трекера")
        verbose_name_plural = _("Хуки трекера")
        permissions = (
            ('import_trackerhookevent', _('Can import tracker hook events')),
            ('export_trackerhookevent', _('Can export tracker hook events')),
        )

    def set_error(self, message: str) -> None:
        self.process_status_result = f"Error: {message}"
        self.status = self.STATUS_ERROR
        self.save()

    def set_success(self, message: str) -> None:
        self.process_status_result = f"Success: {message}"
        self.status = self.STATUS_SUCCESS
        self.save()


class QueueType(models.TextChoices):
    ENROLLMENT = ('enrollments__enrolleduser', _("Зачисление на курс"))
    STUDENTSLOT = ('classrooms__studentslot', _("Слоты для оффлайн-занятий"))


class QueueTypeMixin(models.Model):
    queue_type = models.CharField(_("тип очереди"), max_length=50, choices=QueueType.choices, blank=True)

    class Meta:
        abstract = True


class TrackerQueueQuerySet(ActiveFilterMixin, models.QuerySet):
    pass


class TrackerQueue(TimeStampedModel):
    name = models.CharField(
        verbose_name=_("имя очереди"),
        max_length=255,
        help_text=_("Должно соответствовать очереди в Трекере"),
    )
    display_name = models.CharField(
        verbose_name=_("название очереди"),
        max_length=255,
        blank=True,
        help_text=_("Название этой очереди внутри LMS"),
    )
    issue_type = models.CharField(
        verbose_name=_("тип тикета"),
        max_length=255,
    )
    initial_state = models.CharField(
        verbose_name=_("начальный статус тикета"),
        max_length=255,
        blank=True,
    )
    template = models.ForeignKey(
        TicketTemplate,
        verbose_name=_("шаблон тикета"),
        related_name='tracker_queues',
        null=True,
        blank=True,
        on_delete=models.PROTECT,
    )
    is_active = models.BooleanField(_("активна"), default=True)

    objects = TrackerQueueQuerySet.as_manager()

    history = HistoricalRecords(inherit=True)

    class Meta:
        verbose_name = _("очередь в трекере")
        verbose_name_plural = _("очереди в трекере")

    def __str__(self):
        if self.display_name:
            name = self.display_name
        else:
            name = f"{self.name} | {self.issue_type}"
            if self.initial_state:
                name = f"{name} | {self.initial_state}"

        return name


class TrackerIssue(QueueTypeMixin, TimeStampedModel):
    class Status(models.TextChoices):
        PENDING = ('pending', _("Ожидает"))
        SUCCESS = ('success', _("Обработан"))
        ERROR = ('error', _("Ошибка"))

    queue = models.ForeignKey(
        TrackerQueue,
        verbose_name=_("очередь"),
        related_name='issues',
        on_delete=models.PROTECT,
    )
    issue_key = models.CharField(_("номер тикета"), db_index=True, max_length=255)
    issue_status = models.CharField(_("статус тикета"), max_length=255, blank=True)
    status = models.CharField(_("статус"), max_length=20, choices=Status.choices, default=Status.PENDING)

    @property
    def issue(self) -> str:
        """
        вывод номера тикета (для обратной совместимости)
        :return:
        """
        return self.issue_key

    @property
    def issue_number(self) -> int:
        issue_number = self.issue_key.split('-')[-1]
        return int(issue_number)

    class Meta:
        ordering = ('queue', '-created')
        verbose_name = _("тикет в трекере")
        verbose_name_plural = _("тикеты в трекере")

    def __str__(self):
        return self.issue_key


class TrackerHook(TimeStampedModel, UUIDModel):
    class Status(models.TextChoices):
        PENDING = ('pending', _("В обработке"))
        SUCCESS = ('success', _("Обработан"))
        ERROR = ('error', _("Ошибка"))

    issue = models.ForeignKey(
        TrackerIssue,
        verbose_name=_("тикет"),
        related_name='hooks',
        null=True,
        blank=True,
        on_delete=models.CASCADE,
    )
    data = JSONField(_("данные запроса"), default=dict)
    action = models.CharField(_("действие"), max_length=255, blank=True)
    status = models.CharField(_("статус"), max_length=20, choices=Status.choices, default=Status.PENDING)
    result = models.TextField(_("результат"), blank=True)

    # DEPRACATED
    # поле для связи со старыми хуками
    # после миграции можно удалить
    tracker_hook_event = models.ForeignKey(
        TrackerHookEvent,
        verbose_name=_("старый хук"),
        null=True,
        blank=True,
        on_delete=models.CASCADE,
        related_name='tracker_hooks',
    )

    def __str__(self):
        return f"{self.issue_id} | {self.get_status_display()}"

    class Meta:
        verbose_name = _("Событие из трекера")
        verbose_name_plural = _("События из трекера")

    def set_error(self, message: str = None) -> None:
        self.status = self.Status.ERROR
        self.result = message or ''
        self.save()

    def set_success(self, message: str = None) -> None:
        self.status = self.Status.SUCCESS
        self.result = message or ''
        self.save()


class ClassroomQueue(TrackerQueue):
    cancel_transition = models.CharField(
        verbose_name=_("переход при отмене"),
        help_text=_("Переход, который должен выполниться при отмене заявки сотрудником"),
        max_length=255,
    )

    cancel_comment = models.TextField(
        verbose_name=_("комментарий при отмене"),
        help_text=_("Комментарий, при отмене заявки сотрудником"),
    )

    cancel_tags = JSONField(
        verbose_name=_("теги при отмене"),
        help_text=_("Список тегов, который нужно выставить при отмене заявки сотрудником"),
        default=list,
        blank=True,
    )

    class Meta:
        verbose_name = _("очередь для оффлайн-занятий")
        verbose_name_plural = _("очереди для оффлайн-занятий")


class ClassroomTracker(TimeStampedModel):
    classroom = models.OneToOneField(
        Classroom,
        verbose_name=_("занятие с расписанием"),
        related_name='tracker',
        on_delete=models.CASCADE,
    )
    queue = models.ForeignKey(
        ClassroomQueue,
        verbose_name=_("очередь в трекере"),
        related_name='tracker',
        on_delete=models.CASCADE,
    )

    class Meta:
        verbose_name = _("настройки очереди для оффлайн-занятия")
        verbose_name_plural = _("настройки очередей для оффлайн-занятий")

    def save(self, *args, **kwargs):
        self.validate_unique()
        return super().save(*args, **kwargs)


# DEPRECATED
class ClassroomTrackerQueue(TimeStampedModel):
    classroom = models.ForeignKey(
        Classroom,
        verbose_name=_("занятие с расписанием"),
        on_delete=models.CASCADE,
        # related_name='tracker_queues',
    )
    queue = models.ForeignKey(
        TrackerQueue,
        verbose_name=_("очередь"),
        on_delete=models.PROTECT,
        # related_name='classrooms',
    )

    class Meta:
        verbose_name = _("очередь занятия с расписанием")
        verbose_name_plural = _("очереди занятий с расписанием")

    history = HistoricalRecords()

    def __init__(self, *args, **kwargs):
        warnings.warn('`ClassroomTrackerQueue` model is deprecated', DeprecationWarning)
        super().__init__(*args, **kwargs)

    def __str__(self):
        return f'Classroom: {self.classroom_id}| queue: {self.queue_id}'

    def clean(self):
        other_queues = self.classroom.classroomtrackerqueue_set.all()
        if self.pk is not None:
            other_queues = other_queues.exclude(id=self.pk)
        if other_queues.exists():
            raise ValidationError('already exist')

    def save(self, *args, **kwargs):
        self.full_clean()
        return super().save(*args, **kwargs)


class StudentSlotHookAction(models.TextChoices):
    ACCEPT = "student_slot_accept", _("Подтверждено")
    REJECT = "student_slot_reject", _("Отклонено")


class EnrolledUserHookAction(models.TextChoices):
    ENROLLED = "user_enrolled", _("Заявка подтверждена")
    REJECTED = "user_rejected", _("Заявка отклонена")
    COMPLETED = "user_completed", _("Обучение завершено")


class StudentSlotTrackerIssue(TrackerIssue):
    slot = models.ForeignKey(
        StudentSlot,
        verbose_name=_("запись на слот"),
        on_delete=models.PROTECT,
        related_name='issues',
    )

    class Meta:
        verbose_name = _("тикет на слот в оффлайн-занятии")
        verbose_name_plural = _("тикеты на слоты в оффлайн-занятиях")

    def __str__(self):
        return f'Слот: {self.slot_id}| Тикет: {self.issue_key}'

    def save(self, *args, **kwargs):
        if not self.queue_type:
            self.queue_type = QueueType.STUDENTSLOT
        super().save(*args, **kwargs)


class EnrollmentQueue(TrackerQueue):
    class Meta:
        verbose_name = _("очередь для заявок")
        verbose_name_plural = _("очереди для заявок")


class TrackerQueueAction(TimeStampedModel):
    queue = models.ForeignKey(
        TrackerQueue,
        verbose_name=_("очередь в трекере"),
        on_delete=models.CASCADE,
        related_name='actions',
    )
    issue_status = models.CharField(max_length=255, verbose_name=_("статус тикета в трекере"))
    action = models.CharField(
        max_length=255,
        verbose_name=_("действие"),
        choices=StudentSlotHookAction.choices + EnrolledUserHookAction.choices,
        help_text=_("действие, выполняемое, когда тикет в трекере переходит в указанный статус"),
    )

    class Meta:
        verbose_name = _("действие с тикетами очереди")
        verbose_name_plural = _("действия с тикетами очереди")


class EnrollmentTracker(OrderedModel, TimeStampedModel):
    enrollment = models.ForeignKey(
        Enrollment,
        verbose_name=_("механизм зачисления"),
        related_name='tracker',
        on_delete=models.CASCADE,
    )
    queue = models.ForeignKey(
        EnrollmentQueue,
        verbose_name=_("очередь в трекере"),
        related_name='tracker',
        on_delete=models.CASCADE,
    )
    is_default = models.BooleanField(
        verbose_name=_("очередь по умолчанию"),
        default=False,
    )

    # DEPRACATED
    # поле для связи со старыми очередями
    # после миграции можно удалить
    enrollment_tracker_queue = models.ForeignKey(
        EnrollmentTrackerQueue,
        verbose_name=_("старая очередь на зачисление"),
        null=True,
        blank=True,
        on_delete=models.CASCADE,
        related_name='enrolled_user_tracker_queues',
    )

    def clean(self):
        if self.enrollment.enroll_type != self.enrollment.TYPE_TRACKER:
            raise ValidationError({
                'enrollment': ValidationError(
                    _("Данный механизм зачисления не может содержать очередей в трекере"),
                    code='non_tracker_enrollment'
                ),
            })

    @property
    def is_last_queue(self):
        qs = (
            EnrollmentTracker.objects
            .filter(enrollment_id=self.enrollment_id)
            .exclude(pk=self.pk)
        )

        return not qs.exists()

    def save(self, *args, **kwargs):
        self.clean()
        return super().save(*args, **kwargs)

    def delete(self, *args, **kwargs):
        if self.is_default and not self.is_last_queue:
            raise ValidationError(
                _("Чтобы удалить данную очередь, нужно сначала удалить все остальные очереди"),
                code='default_queue_is_not_last',
            )

        return super().delete(*args, **kwargs)

    order_with_respect_to = 'enrollment'

    class Meta:
        unique_together = (
            ('enrollment', 'queue'),
        )
        verbose_name = _("настройки очереди для заявок")
        verbose_name_plural = _("настройки очередей для заявок")


class EnrolledUserTrackerIssue(TrackerIssue):
    enrolled_user = models.ForeignKey(
        EnrolledUser,
        verbose_name=_("заявка"),
        on_delete=models.PROTECT,
        related_name='issues',
    )
    is_default = models.BooleanField(
        verbose_name=_("тикет в очереди по умолчанию"),
        default=False,
    )

    # DEPRACATED
    # поле для связи со старыми тикетами
    # после миграции можно удалить
    enrollment_tracker_issue = models.ForeignKey(
        EnrollmentTrackerIssue,
        verbose_name=_("старый тикет на зачисление"),
        null=True,
        blank=True,
        on_delete=models.CASCADE,
        related_name='enrolled_user_tracker_issues',
    )

    class Meta:
        verbose_name = _("тикет на заявку")
        verbose_name_plural = _("тикеты на заявки")

    def __str__(self):
        return f'Заявка: {self.enrolled_user_id}| Тикет: {self.issue_key}'

    def save(self, *args, **kwargs):
        if not self.queue_type:
            self.queue_type = QueueType.ENROLLMENT
        super().save(*args, **kwargs)
