import itertools
import logging
import warnings
from datetime import datetime, timedelta
from typing import Iterable, Optional, Tuple

from startrek_client.exceptions import NotFound, StartrekError

from django.contrib.auth import get_user_model
from django.db import transaction
from django.template import loader
from django.utils import timezone
from django.utils.translation import gettext_lazy as _

from lms.classrooms.models import Classroom, StudentSlot
from lms.contrib.startrek.client import startrek_api
from lms.contrib.startrek.settings import STARTREK_EXCLUDE_FIELDS, STARTREK_ROBOT
from lms.courses.models import Course, CourseGroup
from lms.enrollments.models import EnrolledUser, Enrollment
from lms.staff.exceptions import StaffProfileNotFoundForUser
from lms.staff.services import get_course_city_staff_city_map

from .models import (
    ClassroomQueue, ClassroomTrackerQueue, EnrolledUserHookAction, EnrolledUserTrackerIssue, EnrollmentTracker,
    EnrollmentTrackerIssue, EnrollmentTrackerQueue, QueueType, StudentSlotHookAction, StudentSlotTrackerIssue,
    TrackerHook, TrackerHookEvent, TrackerIssue, TrackerQueueAction,
)
from .tickets import EnrollmentFixedTicket, EnrollmentTicket, StudentSlotTicket

logger = logging.getLogger(__name__)
User = get_user_model()


class TrackerStatusException(StartrekError):
    pass


class TrackerTransitionException(StartrekError):
    pass


class TrackerHookError(Exception):
    pass


# DEPRECATED
def get_employee_education_history(user, exclude_ids=None):
    warnings.warn('`get_employee_education_history` is deprecated', DeprecationWarning)

    exclude_ids = exclude_ids or []
    return user.enrolled_to.exclude(
        id__in=exclude_ids,
    ).select_related(
        'course',
        'group',
    ).order_by('-created')


def update_or_create_enrolled_user_ticket(
    enrollment_tracker: EnrollmentTracker,
    enrolled_user: EnrolledUser,
):
    queue = enrollment_tracker.queue
    issue = EnrolledUserTrackerIssue.objects.filter(
        enrolled_user=enrolled_user, queue=queue
    ).first()

    if issue is not None:
        return issue

    find_filter = {
        'runId': enrolled_user.id,
        'queue': queue.name,
    }
    found_issues = startrek_api.issues.find(filter=find_filter)
    startrek_issue = next(iter(found_issues), None)

    if startrek_issue is None:
        # если для очереди задан шаблон - используем шаблонный тикет

        if queue.template:
            ticket = EnrollmentTicket(
                enrolled_user=enrolled_user, queue=queue, is_default_queue=enrollment_tracker.is_default,
            )
        else:
            ticket = EnrollmentFixedTicket(
                enrolled_user=enrolled_user, queue=queue, is_default_queue=enrollment_tracker.is_default,
            )

        startrek_issue = startrek_api.issues.create(**ticket.as_dict())

    tracker_issue = EnrolledUserTrackerIssue(
        enrolled_user=enrolled_user,
        queue=queue,
        issue_key=startrek_issue.key,
        issue_status=startrek_issue.status.key,
        is_default=enrollment_tracker.is_default,
        status=EnrolledUserTrackerIssue.Status.SUCCESS,
    )
    tracker_issue.save()

    return tracker_issue


# DEPRECATED
def update_or_create_enrollment_ticket(
    queue: EnrollmentTrackerQueue,
    enrolled_user: EnrolledUser,
):
    warnings.warn('`update_or_create_enrollment_ticket` is deprecated', DeprecationWarning)

    issue = EnrollmentTrackerIssue.objects.filter(enrolled_user=enrolled_user, queue=queue).first()
    if issue is not None:
        return issue

    find_filter = {
        'runId': enrolled_user.id,
        'queue': queue.name,
    }
    found_issues = startrek_api.issues.find(filter=find_filter)
    startrek_issue = next(iter(found_issues), None)

    if startrek_issue is None:
        startrek_issue = update_or_create_enrollment_slot_ticket(
            queue_name=queue.name,
            access_for_user=queue.is_default,
            queue_description=queue.description,
            issue_type=queue.issue_type,
            summary_template=queue.summary,
            course=enrolled_user.course,
            group=enrolled_user.group,
            user=enrolled_user.user,
            enrolled_user_groups=enrolled_user.groups,
            enrolled_user_id=enrolled_user.id,
            run_id=enrolled_user.id,
            user_answers=enrolled_user.user_answers,
            with_previous_education=True,
            num_hours=enrolled_user.course.num_hours,
            begin_date=enrolled_user.course.begin_date,
            end_date=enrolled_user.course.begin_date,
        )

    issue_number = int(startrek_issue.key.split('-')[-1])

    tracker_issue = EnrollmentTrackerIssue(
        enrolled_user=enrolled_user,
        queue=queue,
        issue_number=issue_number,
        status=startrek_issue.status.key,
    )
    tracker_issue.save()

    return tracker_issue


# DEPRECATED
def receive_ticket_status(tracker_issue: EnrollmentTrackerIssue, force: bool = False):
    warnings.warn('`receive_ticket_status` is deprecated', DeprecationWarning)

    if not tracker_issue.issue_number:
        raise TrackerStatusException('Issue not created')

    old_status = tracker_issue.status
    try:
        new_status = startrek_api.issues[tracker_issue.issue].status.name
    except NotFound:
        raise TrackerStatusException('Issue does not exist')

    if old_status != new_status:
        tracker_issue.status = new_status
        tracker_issue.got_status_from_startrek = timezone.now()
        tracker_issue.save()
        update_enrolled_user_status_by_issue(issue=tracker_issue, force=force)


def get_issue_status(tracker_issue: TrackerIssue):
    """
    Получить статус тикета из трекера и обработать его по настройкам TrackerQueueAction
    """
    if not tracker_issue.issue_key:
        raise TrackerStatusException('Issue not created in tracker')

    old_status = tracker_issue.issue_status
    try:
        new_status = startrek_api.issues[tracker_issue.issue_key].status.name
    except NotFound:
        raise TrackerStatusException('Issue does not exist in tracker')

    if old_status != new_status:
        tracker_issue.issue_status = new_status
        tracker_issue.save()

        tracker_queue_actions = TrackerQueueAction.objects.filter(
            queue_id=tracker_issue.queue_id, issue_status=new_status,
        )
        for tracker_queue_action in tracker_queue_actions:
            process_tracker_action(action=tracker_queue_action.action, issue=tracker_issue)


def update_is_default(instance: EnrollmentTrackerQueue):
    if instance.is_default:
        # Если эта очередь по умолчанию, остальные делаем НЕ по умолчанию
        EnrollmentTrackerQueue.objects.filter(
            enrollment_id=instance.enrollment_id,
        ).exclude(
            id=instance.id,
        ).update(
            is_default=False,
        )


# DEPRECATED
@transaction.atomic()
def update_enrolled_user_status_by_issue(issue: EnrollmentTrackerIssue, force: bool = False) -> None:
    warnings.warn('`update_enrolled_user_status_by_issue` is deprecated', DeprecationWarning)

    if issue.queue.is_default and (not issue.status_processed or force):
        enrolled_user = issue.enrolled_user
        processed = False

        if issue.is_accepted:
            enrolled_user.enroll(_("подтверждено в трекере"))
            processed = True

        elif issue.is_rejected or issue.is_cancelled:
            enrolled_user.reject(_("отклонено в трекере"))
            processed = True

        if processed:
            EnrollmentTrackerIssue.objects.filter(
                id=issue.id,
            ).update(
                status_processed=True,
            )


def receive_ticket_statuses(batch_limit=None, time_limit=None):
    issues = EnrollmentTrackerIssue.objects.filter(
        issue_number__isnull=False,
        status_processed=False,
        queue__tracker_pulling_enabled=True,
    )
    if batch_limit is not None:
        issues = issues[:batch_limit]

    end_time = timezone.now() + timedelta(seconds=time_limit) if time_limit else None

    for issue in issues:
        if end_time and timezone.now() >= end_time:
            break
        try:
            with transaction.atomic():
                locked_issue = EnrollmentTrackerIssue.objects.select_for_update(
                    skip_locked=True,
                ).filter(
                    id=issue.id,
                ).first()
                if locked_issue is not None:
                    receive_ticket_status(locked_issue)
                else:
                    logger.error(f"Issue {issue} locked or does not exist")
        except Exception as exc:
            logger.error(f"Error while receiving ticket {issue} from tracker: {str(exc)}")


def create_tracker_user_enrollment_issues(enrolled_user_id: int) -> None:
    enrolled_user = (
        EnrolledUser.objects
        .select_related(
            'course',
            'course__city',
            'course__study_mode',
            'course__workflow',
            'course__provider',
            'group',
            'group__tutor',
            'group__tutor__user',
            'user',
            'user__staffprofile',
            'user__staffprofile__city',
            'user__staffprofile__office',
            'user__staffprofile__head',
            'user__staffprofile__head__user',
        )
        .get(id=enrolled_user_id)
    )

    if enrolled_user.enrollment and enrolled_user.enrollment.enroll_type == Enrollment.TYPE_TRACKER:
        startrek_issues = []
        for enrollment_tracker in enrolled_user.enrollment.tracker.filter(queue__is_active=True):
            issue = update_or_create_enrolled_user_ticket(
                enrollment_tracker=enrollment_tracker,
                enrolled_user=enrolled_user
            )
            if issue.issue_number:
                startrek_issue = startrek_api.issues[issue.issue]
                startrek_issue.related_issues = [link.object.key for link in startrek_issue.links]
                startrek_issues.append(startrek_issue)

        for startrek_issue1, startrek_issue2 in itertools.combinations(startrek_issues, 2):
            if startrek_issue1.key not in startrek_issue2.related_issues:
                startrek_issue2.links.create(issue=startrek_issue1.key, relationship='relates')


# DEPRECATED
def create_tracker_enrollment_issues(enrolled_user_id: int) -> None:
    warnings.warn('`create_tracker_enrollment_issues` is deprecated', DeprecationWarning)

    enrolled_user = (
        EnrolledUser.objects
        .select_related(
            'course',
            'course__city',
            'course__study_mode',
            'course__workflow',
            'course__provider',
            'group',
            'group__tutor',
            'group__tutor__user',
            'user',
            'user__staffprofile',
            'user__staffprofile__city',
            'user__staffprofile__office',
            'user__staffprofile__head',
            'user__staffprofile__head__user',
        )
        .get(id=enrolled_user_id)
    )

    if enrolled_user.enrollment and enrolled_user.enrollment.enroll_type == Enrollment.TYPE_TRACKER:
        startrek_issues = []
        for queue in enrolled_user.enrollment.tracker_queues.active():
            issue = update_or_create_enrollment_ticket(queue=queue, enrolled_user=enrolled_user)
            if issue.issue_number:
                startrek_issue = startrek_api.issues[issue.issue]
                startrek_issue.related_issues = [link.object.key for link in startrek_issue.links]
                startrek_issues.append(startrek_issue)

        for startrek_issue1, startrek_issue2 in itertools.combinations(startrek_issues, 2):
            if startrek_issue1.key not in startrek_issue2.related_issues:
                startrek_issue2.links.create(issue=startrek_issue1.key, relationship='relates')


def process_tracker_hook(tracker_hook: TrackerHook) -> None:
    """
    Обработка хуков от трекера

    """
    if tracker_hook.status != TrackerHook.Status.PENDING:
        return

    if not isinstance(tracker_hook.data, dict):
        raise TrackerHookError("Некорректный формат data")

    data = tracker_hook.data

    issue_key = data.get("key", None)
    if not issue_key:
        raise TrackerHookError("Ключ тикета не найден")

    with transaction.atomic():
        # находим тикет
        issue: TrackerIssue = (
            TrackerIssue.objects
            .select_for_update(nowait=True)
            .filter(issue_key=issue_key)
            .first()
        )
        if issue is None:
            raise TrackerHookError(f"Тикет {issue_key} не найден")

        tracker_hook.issue = issue

        # определяем действие
        action = data.get("action", None)
        if not action:
            raise TrackerHookError("Действие action не указано")

        tracker_hook.action = action

        # обновляем статус тикета
        issue.issue_status = data.get('status', '')
        issue.status = issue.Status.SUCCESS
        issue.save()

        process_tracker_action(action, issue, tracker_hook)

        # завершаем обработку хука
        tracker_hook.set_success()


def process_tracker_action(action: str, issue: TrackerIssue, hook: Optional[TrackerHook] = None) -> None:
    """
    Обработка действий по хуку из Трекера

    :param action:
    :param issue:
    :param hook:
    :return:
    """
    if issue.queue_type == QueueType.STUDENTSLOT:
        process_classroom_action(action, issue, hook)
    if issue.queue_type == QueueType.ENROLLMENT:
        process_enrolled_user_action(action, issue, hook)


def process_enrolled_user_action(action: str, issue: TrackerIssue, hook: TrackerHook) -> None:
    enrolled_user_tracker_issue = (
        EnrolledUserTrackerIssue.objects
        .select_related(
            'enrolled_user',
            'enrolled_user__enrollment',
            'enrolled_user__enrollment__course',
            'enrolled_user__course',
            'enrolled_user__group',
            'enrolled_user__course_student',
            'enrolled_user__course_student__course',
            'enrolled_user__user',
            'enrolled_user__user__staffprofile',
        )
        .get(pk=issue.id)
    )

    enrolled_user = enrolled_user_tracker_issue.enrolled_user

    # подтвердить заявку
    if action == EnrolledUserHookAction.ENROLLED:
        enrolled_user.enroll(_("подтверждено в трекере ({})").format(action))

    # отклонить заявку
    elif action == EnrolledUserHookAction.REJECTED:
        enrolled_user.reject(_("отклонено в трекере ({})").format(action))

    # обучение завершено
    elif action == EnrolledUserHookAction.COMPLETED:
        enrolled_user.complete(_("завершено в трекере ({})").format(action))

    else:
        raise TrackerHookError(f"Неизвестный action \"{action}\" для EnrolledUser {enrolled_user.pk}")


def process_classroom_action(action: str, issue: TrackerIssue, hook: Optional[TrackerHook] = None) -> None:
    """
    Обработка действий для слотов (оффлайн-занятия)

    :param action:
    :param issue:
    :param hook:
    :return:
    """
    student_slot_issue = (
        StudentSlotTrackerIssue.objects
        .select_related(
            'slot',
            'slot__timeslot',
            'slot__timeslot__course',
            'slot__student',
            'slot__student__course',
        )
        .get(pk=issue.id)
    )
    student_slot = student_slot_issue.slot

    msg = "{} в Трекере"

    # отклонить запись на слот
    if action == StudentSlotHookAction.REJECT and student_slot.status == StudentSlot.StatusChoices.ACCEPTED:
        student_slot.reject(msg.format(StudentSlotHookAction.REJECT.label))

    else:
        raise TrackerHookError(f"Неизвестный action \"{action}\" для StudentSlot {student_slot.pk}")


# DEPRECATED
@transaction.atomic()
def process_tracker_hook_event(tracker_hook_event: TrackerHookEvent, force: bool = False) -> None:
    """
    Обработка хуков от трекера

    :param tracker_hook_event:
    :return:
    """
    warnings.warn("process_tracker_hook_event is deprecated", DeprecationWarning)

    if tracker_hook_event.status != TrackerHookEvent.STATUS_PENDING and not force:
        return

    if not isinstance(tracker_hook_event.request_body, dict):
        tracker_hook_event.set_error('request_body is not a dict')
        return

    issue_key = tracker_hook_event.request_body.get('key')
    if not issue_key:
        tracker_hook_event.set_error('No parameter "key"')
        return

    issue_key_parsed = issue_key.split('-')
    if len(issue_key_parsed) != 2:
        tracker_hook_event.set_error(f'Cannot parse issue key "{issue_key}"')
        return

    queue_name, issue_id = issue_key_parsed
    try:
        issue_id = int(issue_id)
    except (ValueError, TypeError):
        tracker_hook_event.set_error(f'Cannot convert issue id "{issue_id}" to int')
        return

    issue = (
        EnrollmentTrackerIssue.objects
        .select_for_update(nowait=True)
        .filter(issue_number=issue_id, queue__name=queue_name)
        .select_related('queue', 'enrolled_user')
        .first()
    )
    if issue is None:
        tracker_hook_event.set_error(f'Issue "{queue_name}-{issue_id}" not found')
        return

    tracker_hook_event.issue = issue

    if not issue.queue.is_default:
        tracker_hook_event.set_success('Queue is not default')
        return

    issue.status = tracker_hook_event.request_body.get('status', '')
    enrolled_user = issue.enrolled_user

    action = tracker_hook_event.request_body.get('action')
    if action in ['user_enrolled', 'student_renew']:
        enrolled_user.enroll(_("подтверждено в трекере ({})").format(action))

    elif action in ['user_rejected', 'student_expelled']:
        enrolled_user.reject(_("отклонено в трекере ({})").format(action))

    else:
        tracker_hook_event.set_error(f'Unknown action "{action}"')
        return

    issue.status_processed = True
    issue.save()

    tracker_hook_event.set_success('Processed')


# DEPRECATED
def update_or_create_ticket(
    queue_name: str,
    issue_type: str,
    summary_template: str,
    description_template_path: str,
    template_context: dict,
    ticket_fields_data: dict,
):
    """
    Создает тикет в трекере

    :param queue_name:
    :param issue_type:
    :param summary_template:
    :param description_template_path:
    :param template_context:
    :param ticket_fields_data:
    :return:
    """
    warnings.warn("`update_or_create_ticket` is deprecated", DeprecationWarning)

    summary = summary_template.format(**template_context)
    description = loader.render_to_string(
        template_name=description_template_path,
        context=template_context,
    )
    ticket_fields_data['queue'] = queue_name
    ticket_fields_data['summary'] = summary
    ticket_fields_data['type'] = {'name': issue_type}
    ticket_fields_data['description'] = description
    ticket_fields_data = {
        field_name: field_value for field_name, field_value in ticket_fields_data.items()
        if field_name not in STARTREK_EXCLUDE_FIELDS
    }
    return startrek_api.issues.create(**ticket_fields_data)


# DEPRECATED
def update_or_create_enrollment_slot_ticket(
    queue_name: str,
    access_for_user: bool,
    queue_description: str,
    issue_type: str,
    summary_template: str,
    course: Course,
    group: Optional[CourseGroup],
    user: User,
    run_id: int,
    with_previous_education: bool,
    enrolled_user_groups: Optional[str] = None,
    enrolled_user_id: Optional[int] = None,
    user_answers: Optional[list] = None,
    num_hours: Optional[int] = None,
    begin_date: Optional[datetime] = None,
    end_date: Optional[datetime] = None,
    classroom: Optional[Classroom] = None,
):
    warnings.warn("`update_or_create_enrollment_slot_ticket()` is deprecated", DeprecationWarning)

    staff_profile = getattr(user, 'staffprofile', None)
    if staff_profile is None:
        raise StaffProfileNotFoundForUser(user_id=user.id)
    head = staff_profile.head.user if staff_profile.head else None
    hr_partners = [hr_partner.user.username for hr_partner in staff_profile.hr_partners.all()]
    access = list(hr_partners)
    if head:
        access.append(head.username)
    if access_for_user:
        access.append(user.username)
    if STARTREK_ROBOT:
        access.append(STARTREK_ROBOT)

    category = course.categories.first()
    category_name = None if category is None else category.name
    tracker_course_category_id = (
        category.tracker_course_category.tracker_course_category_id
        if (category is not None and hasattr(category, 'tracker_course_category'))
        else None
    )
    course.category_name = category_name

    course_city_name = course.city.name if course.city else ''
    staff_city_name = staff_profile.city.name_ru if staff_profile.city else ''
    has_business_trip: bool = False
    if course.city and staff_profile.city:
        has_business_trip = get_course_city_staff_city_map().get(staff_profile.city_id) != course.city_id

    description_template_path = 'issue/description.html'

    template_context = {
        'description': queue_description,
        'staff_profile': staff_profile,
        'survey_data': user_answers,
        'course_obj': course,
        'group': group,
        'with_previous_education': with_previous_education,
        'has_business_trip': has_business_trip,
        'course_city_name': course_city_name,
        'staff_city_name': staff_city_name,
        'last_name': staff_profile.last_name_ru,
        'first_name': staff_profile.first_name_ru,
        'login': user.username,
        'course': course.name,
        'num_hours': num_hours,
        'begin_date': begin_date,
        'end_date': end_date,
        'classroom': classroom,
    }
    if with_previous_education:
        template_context['enrolled_users'] = get_employee_education_history(
            user=user,
            exclude_ids=[enrolled_user_id],
        )

    start_date = (
        group.begin_date
        if group is not None and group.begin_date is not None
        else course.calc_begin_date
    )
    start_date = start_date.strftime('%Y-%m-%d') if start_date else None

    end_date = (
        group.end_date
        if group is not None and group.end_date is not None
        else course.calc_end_date
    )
    end_date = end_date.strftime('%Y-%m-%d') if end_date else None

    ticket_fields_data = {
        'employee': user.username,
        'department': enrolled_user_groups,
        'office': str(staff_profile.office),
        'head': head.username if head else None,
        'hrbp': hr_partners,
        'edPlace': getattr(staff_profile.city, 'name_ru', None),
        'company': getattr(staff_profile.organization, 'name_ru', None),
        'programName': course.name,
        'employmentForm': getattr(course.study_mode, 'name', None),
        'format': getattr(course.workflow, 'name', None),
        'cityOfEvent': getattr(course.city, 'name', None),
        'provider': getattr(course.provider, 'name', None),
        'start': start_date,
        'end': end_date,
        'access': access,
        'runId': run_id,
        'graphTaskId': course.id,
        'edGroup': group.id if group else None,
        'eventCost': str(course.price),
        'directionOfTraining': tracker_course_category_id,
        'outputGroup': group.id if group else None,
        'startDateAndTime1': start_date,
        'startDateAndTime2': end_date,
    }
    if access_for_user:
        ticket_fields_data['createdBy'] = user.username

    if course.payment_method == Course.PaymentMethodChoices.PERSONAL:
        ticket_fields_data['components'] = f'{course.paid_percent}%'

    if group and group.tutor:
        ticket_fields_data['trainer'] = group.tutor.user.username if group.tutor.is_internal else group.tutor.name

    return update_or_create_ticket(
        queue_name=queue_name,
        issue_type=issue_type,
        summary_template=summary_template,
        description_template_path=description_template_path,
        template_context=template_context,
        ticket_fields_data=ticket_fields_data,
    )


# DEPRECATED
def update_or_create_studentslot_ticket(
    queue: ClassroomTrackerQueue,
    student_slot: StudentSlot,
) -> StudentSlotTrackerIssue:
    """
    Обновляет или создает тикет при записи на слот по оффлайн-занятию

    :param queue:
    :param student_slot:
    :return:
    """
    warnings.warn("update_or_create_studentslot_ticket is deprecated", DeprecationWarning)

    issue = StudentSlotTrackerIssue.objects.filter(slot=student_slot, queue=queue).first()
    if issue is not None:
        return issue

    find_filter = {
        'runId': student_slot.id,
        'queue': queue.queue.queue_name,
    }
    found_issues = startrek_api.issues.find(filter=find_filter)
    startrek_issue = next(iter(found_issues), None)

    if startrek_issue is None:
        begin_date = student_slot.timeslot.begin_date
        end_date = student_slot.timeslot.end_date
        num_hours = (int((end_date - begin_date).total_seconds() // 3600) or None) if end_date else None
        startrek_issue = update_or_create_enrollment_slot_ticket(
            queue_name=queue.queue.queue_name,
            access_for_user=False,
            queue_description=queue.queue.description,
            issue_type=queue.queue.issue_type,
            summary_template=queue.queue.summary,
            course=student_slot.student.course,
            group=student_slot.student.group,
            user=student_slot.student.user,
            run_id=student_slot.id,
            with_previous_education=False,
            num_hours=num_hours,
            begin_date=begin_date,
            end_date=end_date,
            classroom=student_slot.timeslot.classroom,
        )

    issue_number = int(startrek_issue.key.split('-')[-1])

    tracker_issue = StudentSlotTrackerIssue.objects.create(
        slot=student_slot,
        queue=queue,
        issue_number=issue_number,
    )

    return tracker_issue


# DEPRECATED
def create_studentslot_issues(student_slot_id: int):
    """
    Создание тикета на слот оффлайн-занятия

    :param student_slot_id:
    :return:
    """
    warnings.warn("create_studentslot_issues is deprecated", DeprecationWarning)

    student_slot = (
        StudentSlot.objects
        .select_related(
            'timeslot',
            'timeslot__classroom',
            'student',
            'student__group',
            'student__group__tutor',
            'student__group__tutor__user',
        )
        .get(id=student_slot_id)
    )

    startrek_issues = []
    for queue in student_slot.timeslot.classroom.tracker_queues.select_related('queue'):
        issue = update_or_create_studentslot_ticket(queue=queue, student_slot=student_slot)
        if issue.issue_number:
            startrek_issue = startrek_api.issues[issue.issue]
            startrek_issue.related_issues = [link.object.key for link in startrek_issue.links]
            startrek_issues.append(startrek_issue)

    if startrek_issues:
        link_startrek_issues(startrek_issues)


# DEPRECATED
def get_or_create_enrollment_issue(
    queue: EnrollmentTrackerQueue,
    enrolled_user: EnrolledUser,
) -> EnrollmentTrackerIssue:
    """
    Возвращает или создает тикет на зачисление

    :param queue:
    :param enrolled_user:
    :return:
    """
    warnings.warn('`get_or_create_enrollment_issue` is deprecated', DeprecationWarning)

    issue = EnrollmentTrackerIssue.objects.filter(enrolled_user=enrolled_user, queue=queue).first()
    if issue:
        return issue

    filter_params = {
        'runId': enrolled_user.id,
        'queue': queue.name,
    }
    found_startrek_issues = startrek_api.issues.find(filter=filter_params)
    startrek_issue = next(iter(found_startrek_issues), None)

    if startrek_issue is None:
        # если для очереди задан шаблон - используем шаблонный тикет
        if queue.template:
            ticket = EnrollmentTicket(enrolled_user=enrolled_user, queue=queue)
        else:
            ticket = EnrollmentFixedTicket(enrolled_user=enrolled_user, queue=queue)
        startrek_issue = startrek_api.issues.create(**ticket.as_dict())

    issue_number = startrek_issue.key.split('-')[-1]
    issue = EnrollmentTrackerIssue.objects.create(
        enrolled_user=enrolled_user,
        queue=queue,
        issue_number=int(issue_number),
        status=startrek_issue.status.key,
    )
    return issue


def get_or_create_enrolled_user_tracker_issue(
    enrollment_tracker: EnrollmentTracker,
    enrolled_user: EnrolledUser,
) -> EnrolledUserTrackerIssue:
    queue = enrollment_tracker.queue
    issue = EnrolledUserTrackerIssue.objects.filter(
        enrolled_user=enrolled_user,
        queue_id=queue.id,
    ).first()

    if issue:
        return issue

    filter_params = {
        'runId': enrolled_user.id,
        'queue': queue.name,
    }
    found_startrek_issues = startrek_api.issues.find(filter=filter_params)
    startrek_issue = next(iter(found_startrek_issues), None)
    if startrek_issue is None:
        # если для очереди задан шаблон - используем шаблонный тикет

        if queue.template:
            ticket = EnrollmentTicket(
                enrolled_user=enrolled_user, queue=queue, is_default_queue=enrollment_tracker.is_default,
            )
        else:
            ticket = EnrollmentFixedTicket(
                enrolled_user=enrolled_user, queue=queue, is_default_queue=enrollment_tracker.is_default,
            )

        startrek_issue = startrek_api.issues.create(**ticket.as_dict())

    issue = EnrolledUserTrackerIssue.objects.create(
        enrolled_user=enrolled_user,
        queue_id=queue.id,
        is_default=enrollment_tracker.is_default,
        issue_key=startrek_issue.key,
        issue_status=startrek_issue.status.key,
        status=EnrolledUserTrackerIssue.Status.SUCCESS,
    )
    return issue


def link_startrek_issues(startrek_issues: Iterable, relationship: str = 'relates') -> None:
    """
    Связывание тикетов в Трекере

    :param startrek_issues:
    :param relationship:
    :return:
    """
    for first_issue, second_issue in itertools.combinations(startrek_issues, 2):
        if first_issue.key not in second_issue.related_issues:
            second_issue.links.create(issue=first_issue.key, relationship=relationship)


# DEPRECATED
@transaction.atomic()
def process_tracker_enrolled_user_deprecated(enrolled_user_id: int) -> None:
    """
    Обработка заявки при зачислении через трекер

    :param enrolled_user_id:
    :return:
    """
    warnings.warn('`process_tracker_enrolled_user_deprecated` is deprecated', DeprecationWarning)

    # получить заявку и проверить её статус "в ожидании"
    try:
        enrolled_user = (
            EnrolledUser.objects.
            select_related(
                'user',
                'user__staffprofile',
                'user__staffprofile__city',
                'user__staffprofile__office',
                'user__staffprofile__head',
                'user__staffprofile__head__user',
                'user__staffprofile__organization',
                'course',
                'course__city',
                'course__study_mode',
                'course__provider',
                'course__workflow',
                'group',
                'group__tutor',
                'group__tutor__user',
                'survey',
            ).
            select_for_update(of=('self',), skip_locked=True).
            get(
                pk=enrolled_user_id,
                status=EnrolledUser.StatusChoices.PENDING,
            )
        )
    except EnrolledUser.DoesNotExist:
        # если уже обработана - завершить обработку
        logger.info(f"Enrolled user #{enrolled_user_id} already processed")
        return

    enrollment = enrolled_user.enrollment
    if enrollment.enroll_type != enrollment.TYPE_TRACKER:
        logger.warning(f"Enrollment for #{enrolled_user_id} is not for tracker!")
        return

    # перебрать в цикле очереди, установленные для заявки
    startrek_issues = []
    for queue in enrollment.tracker_queues.active():
        # получить или создать тикет для зачисления
        issue = get_or_create_enrollment_issue(queue=queue, enrolled_user=enrolled_user)

        if issue.issue_number:
            startrek_issue = startrek_api.issues[issue.issue]
            startrek_issue.related_issues = [link.object.key for link in startrek_issue.links]
            startrek_issues.append(startrek_issue)

    # слинковать тикеты из разных очередей
    if startrek_issues:
        link_startrek_issues(startrek_issues)


@transaction.atomic()
def process_tracker_enrolled_user(enrolled_user: EnrolledUser) -> None:
    """
    Обработка заявки при зачислении через трекер

    :param enrolled_user:
    :return:
    """
    # получить заявку и проверить её статус "в ожидании"
    enrollment = enrolled_user.enrollment
    if enrollment.enroll_type != enrollment.TYPE_TRACKER:
        logger.warning(f"Enrollment for #{enrolled_user.id} is not for tracker!")
        return

    # перебрать в цикле очереди, установленные для заявки
    startrek_issues = []
    for enrollment_tracker in enrollment.tracker.filter(queue__is_active=True):
        # получить или создать тикет для зачисления
        issue = get_or_create_enrolled_user_tracker_issue(
            enrollment_tracker=enrollment_tracker,
            enrolled_user=enrolled_user,
        )

        if issue.issue_key:
            startrek_issue = startrek_api.issues[issue.issue]
            startrek_issue.related_issues = [link.object.key for link in startrek_issue.links]
            startrek_issues.append(startrek_issue)

    # слинковать тикеты из разных очередей
    if startrek_issues:
        link_startrek_issues(startrek_issues)


def get_or_create_studentslot_issue(queue: ClassroomQueue, student_slot: StudentSlot) -> StudentSlotTrackerIssue:
    """
    Создание/получение тикета при записи студента на слот

    Для оффлайн-занятий

    :param queue:
    :param student_slot:
    :return:
    """
    issue = StudentSlotTrackerIssue.objects.filter(
        slot=student_slot,
        queue=queue,
    ).first()

    if issue:
        return issue

    # проверяем наличие тикета на стороне трекера
    find_filter = {
        'runId': student_slot.id,
        'queue': queue.name,
    }
    found_issues = startrek_api.issues.find(filter=find_filter)
    startrek_issue = next(iter(found_issues), None)

    # если тикет не найден - создаем его
    if startrek_issue is None:
        ticket = StudentSlotTicket(student_slot=student_slot, queue=queue)
        startrek_issue = startrek_api.issues.create(**ticket.as_dict())

    return StudentSlotTrackerIssue.objects.create(
        slot=student_slot,
        queue=queue,
        queue_type=QueueType.STUDENTSLOT,
        issue_key=startrek_issue.key,
        issue_status=startrek_issue.status.key,
    )


def cancel_studentslot_issue(queue: ClassroomQueue, student_slot: StudentSlot) -> Optional[StudentSlotTrackerIssue]:
    """
    Перевод тикета в статус "Отменено" при отмене заявки студентом

    Для оффлайн-занятий

    :param queue:
    :param student_slot:
    :return:
    """
    issue = StudentSlotTrackerIssue.objects.filter(
        slot=student_slot,
        queue=queue,
    ).first()

    if not issue:
        return

    startrek_issue = startrek_api.issues[issue.issue_key]
    current_status = startrek_issue.status

    all_transitions = startrek_issue.transitions.get_all()
    filtered_transitions = [t for t in all_transitions if t.display == queue.cancel_transition]
    transition = next(iter(filtered_transitions), None)

    if not transition:
        msg = f'Для статуса "{current_status.display}" нет перехода "{queue.cancel_transition}"'
        raise TrackerTransitionException(msg)

    transition.execute(comment=queue.cancel_comment, tags=queue.cancel_tags)

    issue.issue_status = transition.to.key
    issue.save()

    return issue


def get_student_slot_queue(student_slot_id: int) -> Tuple[StudentSlot, ClassroomQueue]:
    student_slot = (
        StudentSlot.objects.select_related(
            "timeslot",
            "timeslot__classroom",
            "student",
        )
        .get(id=student_slot_id)
    )

    classroom = student_slot.timeslot.classroom
    tracker = getattr(classroom, 'tracker', None)
    queue = getattr(tracker, 'queue', None)

    return student_slot, queue


@transaction.atomic()
def process_tracker_student_slot_create(student_slot_id: int) -> None:
    """
    Обработка записи на слот (оффлайн-занятия)

    :param student_slot_id:
    :return:
    """
    student_slot, queue = get_student_slot_queue(student_slot_id)

    issue = None
    if queue:
        issue = get_or_create_studentslot_issue(queue, student_slot)
        logger.info(f"StudentSlot({student_slot.id}) Issue({issue}) was successfully created")

    return issue


@transaction.atomic()
def process_tracker_student_slot_cancel(student_slot_id: int) -> None:
    """
    Обработка отмены записи на слот (оффлайн-занятия)

    :param student_slot_id:
    :return:
    """
    student_slot, queue = get_student_slot_queue(student_slot_id)

    issue = None
    if queue:
        issue = cancel_studentslot_issue(queue, student_slot)
        logger.info(f"StudentSlot({student_slot.id}) Issue({issue}) was successfully cancelled")

    return issue
