import logging
import traceback
import warnings
from datetime import timedelta
from typing import Optional

from django.conf import settings
from django.db import transaction
from django.utils import timezone

from config import celery_app as app

from .models import EnrolledUser, Enrollment, TrackerHook, TrackerHookEvent
from .services import (
    TrackerHookError, create_studentslot_issues, create_tracker_enrollment_issues,
    create_tracker_user_enrollment_issues, process_tracker_enrolled_user, process_tracker_enrolled_user_deprecated,
    process_tracker_hook, process_tracker_hook_event, process_tracker_student_slot_cancel,
    process_tracker_student_slot_create, receive_ticket_statuses,
)

log = logging.getLogger(__name__)


@app.task()
def receive_ticket_statuses_task(batch_limit: Optional[int] = None, time_limit: Optional[int] = None) -> None:
    if batch_limit is None:
        batch_limit = settings.CELERY_BATCH_SIZE_DEFAULT
    receive_ticket_statuses(batch_limit=batch_limit, time_limit=time_limit)


@app.task(bind=True)
def create_tracker_enrollment_issues_task(self, enrolled_user_id: int) -> None:
    if settings.ENROLLMENT_QUEUES_MODE in ['new', 'all']:
        try:
            if settings.ENROLLMENT_QUEUES_MODE in ['new', 'all']:
                create_tracker_user_enrollment_issues(enrolled_user_id=enrolled_user_id)

            # deprecated block
            if settings.ENROLLMENT_QUEUES_MODE in ['old', 'all']:
                create_tracker_enrollment_issues(enrolled_user_id=enrolled_user_id)
        except Exception as exc:
            log.exception(f'Error while processing {enrolled_user_id}')
            raise self.retry(exc=exc, max_retries=5, countdown=60)


@app.task()
def create_tracker_enrollment_issues_bulk_task(batch_limit: Optional[int] = None) -> None:
    """
    Периодическая проверка заявок, для которых не было создано тикетов
    """
    max_created_datetime = timezone.now() - timedelta(minutes=settings.ENROLLED_USER_TRACKER_ISSUES_AWAIT_MINUTES)
    queryset = EnrolledUser.objects.filter(
        status=EnrolledUser.StatusChoices.PENDING,
        created__lte=max_created_datetime,
        enrollment__enroll_type=Enrollment.TYPE_TRACKER,
        enrollment__tracker__queue__is_active=True,
        enrollment__tracker__isnull=False,
        issues__isnull=True,
    ).distinct()

    if batch_limit is None:
        batch_limit = settings.CELERY_BATCH_SIZE_DEFAULT

    for enrolled_user in queryset[:batch_limit]:
        log.info('create_tracker_enrollment_issues_bulk for enrolled user: %s', enrolled_user.id)
        create_tracker_enrollment_issues_task.delay(enrolled_user_id=enrolled_user.id)


# DEPRECATED
@app.task(bind=True)
def process_tracker_hook_event_task(self, tracker_hook_event_id: int) -> None:
    warnings.warn("process_tracker_hook_event_task is deprecated", DeprecationWarning)
    try:
        with transaction.atomic():
            tracker_hook_event: TrackerHookEvent = (
                TrackerHookEvent.objects
                .select_for_update(skip_locked=True)
                .get(id=tracker_hook_event_id)
            )

            process_tracker_hook_event(tracker_hook_event=tracker_hook_event)

    except Exception as exc:
        log.error(exc)
        raise self.retry(exc=exc, max_retries=5, countdown=60)


# DEPRECATED
@app.task()
def process_tracker_hook_event_bulk_task(batch_limit: Optional[int] = None) -> None:
    """
    Периодическая проверка необработанных tracker hook event
    """
    warnings.warn("process_tracker_hook_event_bulk_task is deprecated", DeprecationWarning)

    queryset = TrackerHookEvent.objects.filter(status=TrackerHookEvent.STATUS_PENDING)

    if batch_limit is None:
        batch_limit = settings.CELERY_BATCH_SIZE_DEFAULT

    for tracker_hook_event in queryset[:batch_limit]:
        log.info('process_tracker_hook_event_bulk_task for tracker hook event: %s', tracker_hook_event.id)
        process_tracker_hook_event_task.delay(tracker_hook_event_id=tracker_hook_event.id)


@app.task(bind=True)
@transaction.atomic()
def process_tracker_hook_task(self, tracker_hook_id: int) -> None:
    """
    Обработка хуков из Трекера
    """
    try:
        tracker_hook: TrackerHook = (
            TrackerHook.objects
            .select_for_update(skip_locked=True)
            .get(pk=tracker_hook_id, status=TrackerHook.Status.PENDING)
        )

        try:
            process_tracker_hook(tracker_hook)

        except TrackerHookError as exc:
            exc_info = traceback.format_exc()
            msg = f"{exc}\n\n{exc_info}"
            log.error(msg)
            tracker_hook.set_error(msg)

        except Exception as exc:
            log.exception(f"TrackerHook {tracker_hook_id} processing error")
            raise self.retry(exc=exc, max_retries=5, countdown=30)

    except TrackerHook.DoesNotExist:
        log.exception(f"TrackerHook {tracker_hook_id} not found or already processed")


@app.task()
def process_tracker_hook_periodic_task(batch_limit: Optional[int] = None) -> None:
    """
    Периодическая проверка необработанных хуков

    """
    queryset = TrackerHook.objects.filter(status=TrackerHook.Status.PENDING)

    if batch_limit is None:
        batch_limit = settings.CELERY_BATCH_SIZE_DEFAULT

    for obj in queryset[:batch_limit]:
        log.info("process_tracker_hook_periodic_task: %s", obj.id)
        process_tracker_hook_task.delay(tracker_hook_id=obj.id)


# DEPRECATED
@app.task(bind=True)
def create_studentslot_issues_task(self, student_slot_id: int) -> None:
    warnings.warn("create_studentslot_issues_task is deprecated", DeprecationWarning)
    try:
        create_studentslot_issues(student_slot_id=student_slot_id)
    except Exception as exc:
        log.error(exc)
        raise self.retry(exc=exc, max_retries=5, countdown=60)


@app.task(bind=True)
def process_tracker_enrolled_user_task(self, enrolled_user_id: int) -> None:
    if settings.ENROLLMENT_QUEUES_MODE in ['new', 'all']:
        try:
            with transaction.atomic():
                enrolled_user = (
                    EnrolledUser.objects
                    .select_for_update(skip_locked=True)
                    .get(id=enrolled_user_id)
                )
                process_tracker_enrolled_user(enrolled_user=enrolled_user)
        except Exception as exc:
            log.error(exc)
            raise self.retry(exc=exc, max_retries=5, countdown=60)

    # deprecated block
    if settings.ENROLLMENT_QUEUES_MODE in ['old', 'all']:
        try:
            process_tracker_enrolled_user_deprecated(enrolled_user_id=enrolled_user_id)
        except Exception as exc:
            log.exception(f'Error while processing {enrolled_user_id}')
            raise self.retry(exc=exc, max_retries=5, countdown=60)


@app.task(bind=True)
def process_tracker_student_slot_create_task(self, student_slot_id: int) -> None:
    """
    Обработка записи на слот (оффлайн-занятия)
    """
    try:
        process_tracker_student_slot_create(student_slot_id=student_slot_id)
    except Exception as exc:
        log.exception(f'Error while processing {student_slot_id}')
        raise self.retry(exc=exc, max_retries=5, countdown=50)


@app.task(bind=True)
def process_tracker_student_slot_cancel_task(self, student_slot_id: int) -> None:
    """
    Обработка отмены записи на слот (оффлайн-занятия)
    """
    try:
        process_tracker_student_slot_cancel(student_slot_id=student_slot_id)
    except Exception as exc:
        log.exception(f'Error while processing {student_slot_id}')
        raise self.retry(exc=exc, max_retries=5, countdown=50)
