import json
from typing import List, Optional, Union

from django.template import Context, Template, loader
from django.utils.functional import cached_property

from lms.classrooms.models import StudentSlot
from lms.contrib.startrek.settings import STARTREK_EXCLUDE_FIELDS, STARTREK_ROBOT
from lms.enrollments.models import EnrolledUser
from lms.staff.models import StaffProfile
from lms.staff.services import get_course_city_staff_city_map
from lms.tracker.models import EnrollmentTrackerQueue, TicketFieldTemplate, TrackerQueue

from .models import TrackerCourseCategory


def render_string(template_string: str, context: dict) -> str:
    template_string = "{% load ticket_tags %}" + template_string
    context = Context(context)
    data = Template(template_string).render(context)
    return str(data).strip()


class BaseTicket:
    """
    Базовый класс для создания тикетов

    """
    exclude_fields: Optional[list] = None

    def __init__(self, queue: Union[EnrollmentTrackerQueue, TrackerQueue], is_default_queue: bool = False):
        self.queue = queue
        if not hasattr(self.queue, 'is_default'):
            self.queue.is_default = is_default_queue

    def get_context_data(self) -> dict:
        """
        Контекст для передачи данных, необходимых для тикета.

        Здесь только формируется контекст, для получения использовать `context_data`!
        :return:
        """
        raise NotImplementedError()

    @cached_property
    def context_data(self):
        """
        Данные из контекста.

        :return:
        """
        if not hasattr(self, '_context_data'):
            self._context_data = self.get_context_data()

        return self._context_data

    def render_template(self, template: str) -> str:
        """
        Рендеринг шаблона из строки

        :param template:
        :return:
        """
        tmpl = Template(template)
        context = Context(self.get_context_data())
        return tmpl.render(context)

    def filter_fields(self, fields: dict) -> dict:
        """
        Фильтрация полей тикета

        :param fields:
        :return:
        """
        exclude = self.exclude_fields or []
        return {key: val for key, val in fields.items() if key not in exclude}

    def get_access_for(self, context) -> list:
        """
        Возвращает список логинов для доступа к тикету

        :return:
        """
        return []

    def get_fields(self, context) -> dict:
        """
        Возвращает словарь полей для тикета

        :return:
        """
        return {}

    def get_summary(self, context) -> str:
        """
        Возвращает заголовок (summary) тикета

        :return:
        """
        raise NotImplementedError()

    def get_description(self, context) -> Optional[str]:
        """
        Возвращает описание (description) тикета

        :return:
        """
        return None

    def as_dict(self) -> dict:
        """
        Возвращает набор данных для создания тикета

        :return:
        """
        context = self.context_data
        ticket = self.filter_fields(self.get_fields(context))

        ticket['queue'] = self.queue.name
        ticket['summary'] = self.get_summary(context)

        description = self.get_description(context)
        if description:
            ticket['description'] = description

        access = self.get_access_for(context)
        if access:
            ticket['access'] = access

        return ticket


class TicketTemplateMixin:
    """
    Миксин для шаблонного рендеринга

    """
    def __init__(self: BaseTicket, *args, **kwargs):
        super().__init__(*args, **kwargs)

        template = getattr(self.queue, 'template', None)
        if template is None:
            raise ValueError('Template not set')

        self.template = template

    def get_template_fields(self):
        return self.template.fields.active()

    def get_summary(self, context) -> str:
        return render_string(self.template.summary, context)

    def get_description(self, context) -> Optional[str]:
        description = self.template.description
        return render_string(description, context) if description else None

    def get_fields(self, context) -> dict:
        fields = {}
        for field in self.get_template_fields():  # type: TicketFieldTemplate
            data = render_string(field.value, context)
            # если включен allow_json, преобразуем в json
            if field.allow_json:
                try:
                    data = json.loads(data)
                except ValueError:
                    data = None

            # если поле не пустое, или включен allow_null - выводим
            if data or field.allow_null:
                fields[field.field_name] = data

        return fields


class EnrollmentFixedTicket(BaseTicket):
    """
    Класс тикета на зачисление (старый шаблон)

    Пример использования:

    ticket = EnrollmentFixedTicket(queue, enrolled_user)
    startrek_api.issues.create(**ticket.as_dict())

    """
    description_template_path = 'new_issue/enrollment.html'
    exclude_fields = STARTREK_EXCLUDE_FIELDS

    def __init__(self, enrolled_user: EnrolledUser, *args, **kwargs):
        self.enrolled_user = enrolled_user
        super().__init__(*args, **kwargs)

    def check_business_trip(self, course, staff_profile) -> bool:
        """
        Проверяет, нужна ли пользователю командировка

        :param course:
        :param staff_profile:
        :return:
        """
        has_bt = False
        if course.city_id and staff_profile.city_id:
            has_bt = get_course_city_staff_city_map().get(staff_profile.city_id) != course.city_id

        return has_bt

    def get_education_history(self, user, exclude_ids=None):
        """
        Возвращает историю предыдущего обучения пользователя

        :param user:
        :param exclude_ids: фильтр по id заявок
        :return:
        """
        exclude_ids = exclude_ids or []
        return user.enrolled_to.exclude(
            id__in=exclude_ids,
        ).select_related(
            'course',
            'group',
        ).order_by('-created')

    def get_context_data(self) -> dict:
        user = self.enrolled_user.user
        staff_profile: StaffProfile = getattr(user, 'staffprofile', None)
        if not staff_profile:
            raise ValueError('No user staff_profile found!')

        course = self.enrolled_user.course
        group = self.enrolled_user.group

        start_date = group.begin_date if group and group.begin_date else course.calc_begin_date
        start_date = start_date and start_date.strftime('%Y-%m-%d')

        end_date = group.end_date if group and group.end_date else course.calc_end_date
        end_date = end_date and end_date.strftime('%Y-%m-%d')

        return {
            'queue': self.queue,
            'issue_type': {'name': self.queue.issue_type},
            'enrolled_user': self.enrolled_user,
            'user': user,
            'staff_profile': staff_profile,
            'course': course,
            'course_category': course.categories.first(),
            'group': group,
            'hr_partners': list(staff_profile.hr_partners.values_list('user__username', flat=True)),
            'start_date': start_date,
            'end_date': end_date,
            'has_business_trip': self.check_business_trip(course, staff_profile),
            'education_history': self.get_education_history(user, exclude_ids=[self.enrolled_user.pk]),
            'tracker_course_categories': {
                str(category.id): category
                for category in TrackerCourseCategory.objects.select_related('category')
            },
        }

    def get_fields(self, context) -> dict:
        queue = context['queue']
        enrolled_user = context['enrolled_user']
        user = context['user']
        staff_profile = context['staff_profile']
        hr_partners = context['hr_partners']
        course = context['course']
        course_category = context['course_category']
        group = context['group']
        start_date = context['start_date']
        end_date = context['end_date']

        head = getattr(staff_profile, 'head', None)

        fields = {
            'type': {'name': queue.issue_type},
            'employee': user.username,
            'department': enrolled_user.groups,
            'runId': enrolled_user.id,
            'graphTaskId': course.id,
            'office': str(staff_profile.office),

            # {{ staff_profile.head.username }}
            'head': head and head.user.username,
            'hrbp': hr_partners,

            # {{ staff_profile.city.name_ru }}
            'edPlace': getattr(staff_profile.city, 'name_ru', None),
            'edGroup': group and group.id,
            'outputGroup': group and group.id,
            'company': getattr(staff_profile.organization, 'name_ru', None),

            'programName': course.name,
            'eventCost': str(course.price),
            'employmentForm': getattr(course.study_mode, 'name', None),
            'format': getattr(course.workflow, 'name', None),
            'cityOfEvent': getattr(course.city, 'name', None),
            'provider': getattr(course.provider, 'name', None),
            'start': start_date,
            'startDateAndTime1': start_date,
            'end': end_date,
            'startDateAndTime2': end_date,
        }

        if queue.is_default:
            fields['createdBy'] = user.username

        if course.payment_method == course.PaymentMethodChoices.PERSONAL:
            fields['components'] = f"{course.paid_percent}%"

        if group and group.tutor:
            fields['trainer'] = str(group.tutor)

        tracker_course_category_id = (
            course_category and
            hasattr(course_category, 'tracker_course_category') and
            course_category.tracker_course_category.tracker_course_category_id
        )
        fields['directionOfTraining'] = tracker_course_category_id or None

        return fields

    def get_access_for(self, context) -> Optional[List[str]]:
        queue = context['queue']
        user = context['user']
        staff_profile = context['staff_profile']
        hr_partners = context['hr_partners']

        access = list(hr_partners)

        head = staff_profile.head and staff_profile.head.user
        if head:
            access.append(head.username)

        if queue.is_default:
            access.append(user.username)

        if STARTREK_ROBOT:
            access.append(STARTREK_ROBOT)

        return access

    def get_summary(self, context) -> str:
        queue = context['queue']
        course = context['course']
        staff_profile = context['staff_profile']
        user = context['user']

        # отдельный контекст для обратной совместимости
        summary_context = {
            'course': course.name,
            'first_name': staff_profile.first_name_ru,
            'last_name': staff_profile.last_name_ru,
            'login': user.username,
        }

        return queue.summary.format(**summary_context)

    def get_description(self, context) -> Optional[str]:
        return loader.render_to_string(
            template_name=self.description_template_path,
            context=context,
        )


class EnrollmentTicket(TicketTemplateMixin, EnrollmentFixedTicket):
    """
    Класс для тикетов на зачисление

    Использует шаблоны тикетов (TicketTemplate)
    """
    pass


class StudentSlotTicket(TicketTemplateMixin, BaseTicket):
    """
    Класс для тикетов на слоты (по оффлайн-занятиям)

    Использует шаблоны тикетов (TicketTemplate)
    """
    exclude_fields = STARTREK_EXCLUDE_FIELDS

    def __init__(self, student_slot: StudentSlot, *args, **kwargs):
        self.student_slot = student_slot
        super().__init__(*args, **kwargs)

    def get_context_data(self) -> dict:
        student = self.student_slot.student
        timeslot = self.student_slot.timeslot

        start_date = timeslot.begin_date and timeslot.begin_date.strftime("%Y-%m-%d")
        end_date = timeslot.end_date and timeslot.end_date.strftime("%Y-%m-%d")
        num_hours = (
            int((timeslot.end_date - timeslot.begin_date).total_seconds() // 3600) or None
        ) if timeslot.end_date else None

        return {
            "queue": self.queue,
            "course": timeslot.course,
            "classroom": timeslot.classroom,
            "student_slot": self.student_slot,
            "group": student.group,
            "user": student.user,
            "start_date": start_date,
            "end_date": end_date,
            "num_hours": num_hours,
        }
