from drf_spectacular.utils import extend_schema

from django.utils.translation import gettext

from rest_framework.permissions import IsAuthenticated

from lms.core.views.viewsets import APIModelViewSet
from lms.courses.models import CourseStudent

from ..models import EnrollmentTrackerIssue, StudentSlotTrackerIssue
from ..serializers import EnrollmentTrackerIssueListSerializer, StudentSlotTrackerIssueSerializer


class EnrolledUserTrackerIssueViewSet(APIModelViewSet):
    serializer_class = EnrollmentTrackerIssueListSerializer
    queryset = (
        EnrollmentTrackerIssue.objects
        .active()
        .select_related('queue')
        .filter(queue__is_default=True)
    )
    permission_classes = [
        IsAuthenticated,
    ]
    pagination_class = None
    lookup_url_kwarg = 'pk'
    lookup_field = 'enrolled_user_id'

    def get_queryset(self):
        queryset = super().get_queryset()
        if self.action == 'list':
            timeslot_id = self.kwargs['pk']
            filter_kwargs = {self.lookup_field: timeslot_id}
            queryset = queryset.filter(**filter_kwargs)

        user = getattr(self.request, 'user', None)
        if user:
            return queryset.filter(enrolled_user__user=user)

        return queryset.none()

    @extend_schema(
        summary=gettext("Список тикетов из трекера")
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)


class StudentSlotTrackerIssueViewSet(APIModelViewSet):
    serializer_class = StudentSlotTrackerIssueSerializer
    queryset = (
        StudentSlotTrackerIssue.objects
        .filter(slot__student__status=CourseStudent.StatusChoices.ACTIVE)
        .select_related('queue')
    )

    permission_classes = [
        IsAuthenticated,
    ]

    pagination_class = None
    lookup_url_kwarg = 'pk'
    lookup_field = 'slot__timeslot_id'

    def get_queryset(self):
        queryset = super().get_queryset()
        if self.action == 'list':
            timeslot_id = self.kwargs['pk']
            filter_kwargs = {self.lookup_field: timeslot_id}
            queryset = queryset.filter(**filter_kwargs)

        user = getattr(self.request, 'user', None)
        if user:
            return queryset.filter(slot__student__user=user)

        return queryset.none()

    @extend_schema(
        summary=gettext("Список тикетов из трекера")
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)
