from model_utils.models import TimeStampedModel

from django.contrib.auth.models import AbstractUser
from django.contrib.auth.models import Group as BaseGroup
from django.contrib.auth.models import Permission
from django.db import models
from django.utils.translation import gettext_lazy as _

from lms.core.models.mixins import ActiveFilterMixin, ActiveModelMixin


class User(AbstractUser):
    yauid = models.BigIntegerField(_('UID'), null=True, blank=True, db_index=True)

    YAUID_FIELD = 'yauid'
    REQUIRED_FIELDS = ['yauid']

    class Meta(AbstractUser.Meta):
        permissions = [
            ('can_view_in_lab', _("Can view in laboratory")),
            ('can_edit_in_lab', _("Can edit in laboratory")),
        ]

    def get_staff_email(self):
        return self.email if self.email else f'{self.username}@yandex-team.ru'


class UserForSupport(User):
    class Meta:
        proxy = True
        verbose_name = _('[Для поддержки] Пользователь')
        verbose_name_plural = _('[Для поддержки] Пользователи')


class Group(BaseGroup):
    class Meta:
        proxy = True
        verbose_name = _('служебная группа')
        verbose_name_plural = _('служебные группы')


class PermissionPreset(models.Model):
    name = models.CharField(max_length=255, verbose_name=_("название набора разрешений"), unique=True)
    permissions = models.ManyToManyField(
        to=Permission,
        verbose_name=_("разрешения"),
        related_name='presets',
        blank=True,
    )

    def __str__(self):
        return self.name

    class Meta:
        verbose_name = _("набор разрешений")
        verbose_name_plural = _("наборы разрешений")


class ServiceAccountQuerySet(ActiveFilterMixin, models.QuerySet):
    pass


class ServiceAccount(ActiveModelMixin, TimeStampedModel):
    tvm_id = models.PositiveIntegerField(_("ID TVM-приложения"), unique=True)
    name = models.CharField(_("название"), max_length=255)
    description = models.TextField(_("описание"), blank=True)

    objects = ServiceAccountQuerySet.as_manager()

    class Meta:
        verbose_name = _("Сервисный аккаунт")
        verbose_name_plural = _("Сервисные аккаунты")

    def __str__(self):
        return f'tvm_id: {self.tvm_id} name: {self.name}'
