# coding: utf-8
from __future__ import unicode_literals

import logging
import requests

from django.conf import settings

from login_linker import exceptions

from ids.helpers.oauth import get_token


logger = logging.getLogger(__name__)


class OAuthController(object):

    def __init__(self, observer_uid, int_login):
        self.observer_uid = observer_uid
        self.int_login = int_login

    @property
    def redirect_url(self):
        return '{url}&state={login}'.format(
            url=settings.OAUTH_GET_TOKEN_URL,
            login=self.int_login
        )

    def _get_access_token(self, submit_code):
        url = '{url}token'.format(url=settings.OAUTH_URL)
        data = {
            'grant_type': 'authorization_code',
            'code': submit_code,
            'client_id': settings.EXTERNAL_OAUTH_ID,
            'client_secret': settings.EXTERNAL_OAUTH_SECRET,
        }
        response = requests.post(url, data, timeout=2)
        response = response.json()
        if 'error' in response:
            logger.error(
                'User: %s, error trying to retrieve token by code: %s',
                self.int_login,
                response,
            )
            raise exceptions.OAuthException
        return response['access_token']

    def _get_external_credentials(self, access_token):
        """
        Делаем запрос в Oauth за внешним логином
        """
        url = settings.YANDEX_LOGIN_INFO_URL
        params = {'format': 'json', 'oauth_token': access_token}
        try:
            result = requests.get(url, params=params, timeout=1).json()
            return result.get('login'), result.get('id')
        except Exception:
            logger.error(
                'User %s: error trying to retrieve external login',
                self.int_login
            )
            raise exceptions.OAuthException

    def staff_request(self, url, data):
        token = get_token(
            uid=self.observer_uid,
            oauth_id=settings.OAUTH_ID,
            oauth_secret=settings.OAUTH_SECRET,
        )

        result = requests.post(
            settings.STAFF_HOST + url,
            data=data,
            timeout=settings.STAFF_TIMEOUT,
            allow_redirects=False,
            headers={'Authorization': 'OAuth %s' % token}
        )

        if result.status_code >= 300:
            logger.info(
                'Staff respond: status=%s, content=%s',
                result.status_code,
                result.content,
            )

        if result.status_code == 400:
            raise exceptions.StaffInvalidInternalLogin
        elif result.status_code == 403:
            raise exceptions.StaffForbidden
        elif result.status_code == 404:
            raise exceptions.StaffInvalidApp
        elif result.status_code >= 300:
            raise exceptions.StaffError

        return result

    def process_complete(self, oauth_data):
        """
        Обрабатываем успешно прошедшую аутентификацию:
         - получаем код,
            который меняем на токен авторизированного пользователя,
         - создаём/активируем запись ExternalLogin
        """
        if 'error' in oauth_data:
            raise exceptions.OAuthException()

        submit_code = oauth_data.get('code')
        access_token = self._get_access_token(submit_code)

        ext_login, ext_uid = self._get_external_credentials(access_token)
        logger.info('Activating %s (%s) for internal login %s', ext_login, ext_uid, self.int_login)

        return self.staff_request(
            '/oauth/activate/',
            {
                'int_login': self.int_login,
                'ext_login': ext_login,
                'ext_uid': ext_uid,
            }
        )

    def unlink(self):
        logger.info('Deactivating for internal login %s', self.int_login)
        return self.staff_request(
            '/oauth/deactivate/',
            {'int_login': self.int_login}
        )
