import numbers
import os
import attr

from library.python import resource
from attr.validators import instance_of, optional
from django.utils.functional import cached_property

from .base import BaseDto, CustomSerialization


def skip_value(attribute, value):
    return attribute.name != 'value' and value is not None


def skip_blank_string(attribute, value):
    return value != ''


def skip_string_values(attribute, value):
    return attribute.name not in ('color', 'action', 'strike', 'hint') or value is not None


class TypeMetaClass(type):
    def __new__(mcs, name, bases, attrs):
        attrs.setdefault('type', name.lower())
        return super().__new__(mcs, name, bases, attrs)


@attr.s
class Type(BaseDto, metaclass=TypeMetaClass):
    def to_dict(self, **kwargs):
        return {'type': self.type, **super().to_dict(**kwargs, recurse=False)}


@attr.s
class List(Type):
    value = attr.ib(validator=instance_of(list))
    separator = attr.ib(default=chr(160), validator=instance_of(str))
    ttl = attr.ib(default=10800, validator=instance_of(int))
    action = attr.ib(default=None, validator=optional(instance_of(dict)))
    completed = attr.ib(default=True, validator=instance_of(bool))

    def to_dict(self, **kwargs):
        return {'value': [v.to_dict() for v in self.value], **super().to_dict(filter=skip_value)}


@attr.s
class String(Type):
    value = attr.ib(validator=instance_of(str), converter=str)
    color = attr.ib(default=None, validator=optional(instance_of(str)))
    action = attr.ib(default=None, validator=optional(instance_of(dict)))
    strike = attr.ib(default=None, validator=optional(instance_of(bool)))
    hint = attr.ib(default=None, validator=optional(instance_of(str)))

    def to_dict(self, **kwargs):
        return super().to_dict(**kwargs, filter=skip_string_values)


@attr.s
class Number(Type):
    value = attr.ib(validator=instance_of(numbers.Number))
    format = attr.ib(default='%d', validator=instance_of(str))


@attr.s
class Image(Type):
    src = attr.ib()
    width = attr.ib(validator=instance_of(int), default=32)
    height = attr.ib(validator=instance_of(int), default=32)
    text = attr.ib(validator=instance_of(str), default='')

    @src.validator
    def src_validate(self, attribute, value):
        if not any(isinstance(value,
                              cls) for cls in (str,
                                               BinaryResource)):
            raise TypeError('got wrong class for src {}'.format(
                type(value)
            ))


@attr.s
class User(Type):
    login = attr.ib(validator=instance_of(str))
    title = attr.ib(validator=instance_of(str), default='')

    def to_dict(self, **kwargs):
        return super().to_dict(**kwargs, filter=skip_blank_string)


@attr.s
class Inflect(Type):
    name = attr.ib(validator=instance_of(str))
    last_name = attr.ib(validator=instance_of(str), default='')
    is_dismissed = attr.ib(validator=instance_of(bool), default=False)

    def to_dict(self, **kwargs):
        return super().to_dict(**kwargs, filter=skip_blank_string)


@attr.s
class Datetime(Type):
    """Represents datetime in ISO-8601"""
    value = attr.ib(validator=instance_of(str))

    @classmethod
    def from_datetime(cls, dt):
        return cls(value=dt.isoformat())


class BinaryResource(CustomSerialization):
    base_path = 'intranet/magiclinks/src/links/workers/resources/'

    def __init__(self, icon_name, path):
        self.icon_abs_path = self.path_to_icon(icon_name, path)

    def path_to_icon(self, icon_name, path):
        folder_name = os.path.basename(path).split('.')[0]
        _, instance = os.path.split(os.path.dirname(path))
        result = os.path.join(self.base_path, instance, folder_name, icon_name)
        if not resource.resfs_src(result.encode('utf-8'), resfs_file=True):
            raise ValueError('"{0}" is not a file'.format(result))
        return result

    @cached_property
    def resource(self):
        return resource.resfs_read(self.icon_abs_path).decode('utf-8')
        return open(self.icon_abs_path, 'r').read()

    def serialize(self):
        return self.resource

