import asyncio
import aiohttp
import socket
from concurrent.futures import ThreadPoolExecutor

from django.conf import settings

from .dto import Result


class Runner:
    """Class that represents top-level business logic"""

    def __init__(self, request, workers_manager, url_parser,
                 pool_executor_class=ThreadPoolExecutor):
        """
        :type request: WSGIRequest
        :type workers_manager: WorkersManager
        :type url_parser: UrlsParser
        :type pool_executor_class: type
        """
        self.request = request
        self.workers_manager = workers_manager
        self.url_parser = url_parser
        self.pool_executor_class = pool_executor_class

    async def run(self, urls):
        """
        :type urls: list[str]
        :rtype: Result
        """
        executor = self.get_executor()
        async with self.get_session() as session:
            workers = self.workers_manager.create_workers(request=self.request,
                                                          url_parser=self.url_parser,
                                                          urls=urls,
                                                          executor=executor,
                                                          session=session,
                                                          )

            result_obj = Result(data={})
            tasks = [asyncio.ensure_future(worker.get())
                     for worker in workers]

            for future in asyncio.as_completed(tasks):
                result = await future
                result_obj.data.update(self.result_to_dict(result))
                if any((response and not response.completed)
                       for url, response in result.data.items()):
                    result_obj.completed = False

        return result_obj

    def result_to_dict(self, result):
        return {url: response
                for url, response in result.data.items()
                if response
                }

    def get_executor(self, max_workers=5):
        """
        :type max_workers: int
        :rtype: concurrent.futures.Executor
        """
        return self.pool_executor_class(max_workers=max_workers)

    def get_session(self):
        return aiohttp.ClientSession(**self.get_session_params())

    def get_session_params(self):
        return dict(
            connector=aiohttp.TCPConnector(
                verify_ssl=settings.SSL_VERIFY,
                family=socket.AF_INET6,
            ),
        )
