# coding: utf-8
import asyncio
import functools

from django.conf import settings

from tvm2 import TVM2


class TVM2Error(Exception):
    pass


def get_tvm2_client():
    return TVM2(
        client_id=settings.TVM_CLIENT_ID,
        secret=settings.TVM_SECRET,
        blackbox_client=settings.BLACKBOX_CLIENT,
        destinations=settings.TVM2_DESTINATIONS,
    )


def get_service_ticket_sync(client_id):
    tvm2_client = get_tvm2_client()
    service_tickets = tvm2_client.get_service_tickets(client_id)
    service_ticket = service_tickets.get(client_id)
    if not service_ticket:
        raise TVM2Error('Can\'t get service ticket for client_id={}'.format(client_id))
    return service_ticket


async def get_service_ticket(client_id, executor):
    tvm2_client = get_tvm2_client()

    loop = asyncio.get_event_loop()
    get_service_tickets = functools.partial(
        tvm2_client.get_service_tickets,
        client_id,
    )
    task = loop.run_in_executor(executor, get_service_tickets)
    service_tickets = await task

    service_ticket = service_tickets.get(client_id)
    if not service_ticket:
        raise TVM2Error('Can\'t get service ticket for client_id={}'.format(client_id))
    return service_ticket


async def get_user_ticket(user_ip, server_host, oauth_token, session_id, executor):
    tvm2_client = get_tvm2_client()

    loop = asyncio.get_event_loop()
    get_user_ticket = functools.partial(
        tvm2_client.get_user_ticket,
        user_ip,
        server_host,
        oauth_token,
        session_id,
    )
    task = loop.run_in_executor(executor, get_user_ticket)
    user_ticket = await task

    if not user_ticket:
        raise TVM2Error('Can\'t get user ticket from blackbox')

    return user_ticket
