import logging

from importlib import import_module
from collections import OrderedDict
from django.conf import settings
from django.utils.functional import cached_property

from .match_options import compile_match_options

logger = logging.getLogger(__name__)


class UnknownWorkerClass(ValueError):
    def __init__(self, worker_class_file):
        self.worker_class_file = worker_class_file

    def __str__(self):
        return 'Unknown worker class: {}'.format(self.worker_class_file)


def get_module_path(instance_type, file_name):
    return 'intranet.magiclinks.src.links.workers.{}.{}'.format(
        instance_type, file_name,
    )


def get_worker(instance_type, worker_class_file):
    path = get_module_path(instance_type, worker_class_file)
    try:
        worker_module = import_module(path)
        return getattr(worker_module, 'Worker')
    except (AttributeError, ImportError,):
        raise UnknownWorkerClass(worker_class_file)


def import_worker_class(worker_class_file):
    instance_type = settings.INSTANCE_TYPE

    try:
        worker = get_worker(instance_type, worker_class_file)
    except UnknownWorkerClass:
        logger.info('Getting default worker for "%s"', worker_class_file)
        worker = get_worker('default', worker_class_file)

    return worker


class WorkersManager:

    def __init__(self, workers):
        self.workers = workers

    @cached_property
    def workers_classes(self):
        return OrderedDict(((worker_class_file, import_worker_class(worker_class_file))
                            for worker_class_file in self.workers))

    def create_workers(self, request, url_parser, urls, executor, session):
        """
        Create workers, update them
        with parsed urls and return
        workers with urls

        :type request: WSGIRequest
        :type url_parser: UrlsParser
        :type urls: list
        :type executor ThreadPoolExecutor
        :rtype: list[WorkerInterface]
        """
        workers_map = self.create_workers_map(request=request,
                                              executor=executor,
                                              session=session,
                                              )
        parsed_urls = url_parser.parse_urls(urls, self.compiled_workers)
        return self.populate_workers(workers_map, parsed_urls)

    def populate_workers(self, workers_map, parsed_urls):
        for parsed_url in parsed_urls:
            worker_class = parsed_url.worker_class_file
            workers_map[worker_class].add_url_object(parsed_url)
        workers_with_urls = self.get_workers_with_urls(workers_map)
        return workers_with_urls

    def create_workers_map(self, request, executor, session):
        """
        Returns a map of Worker objects
        :type request: WSGIRequest
        :type executor ThreadPoolExecutor
        :rtype: dict[worker_name: WorkerInterface]
        """
        workers = {worker_class_file: self.workers_classes[worker_class_file](request=request,
                                                                              executor=executor,
                                                                              session=session,
                                                                              )
                   for worker_class_file in self.workers}

        return workers

    @cached_property
    def compiled_workers(self):
        return OrderedDict(((worker, compile_match_options(worker_class))
                            for worker, worker_class
                            in self.workers_classes.items()))

    def get_workers_with_urls(self, workers):
        return [worker for worker in workers.values() if len(worker.url_objects) > 0]


workers_manager = WorkersManager(settings.WORKERS)
