import json
import time
import logging
import asyncio
from django_tools_log_context import request_context, request_profiler

from django.http import JsonResponse
from django.utils.decorators import method_decorator
from django.views import View
from django.views.decorators.csrf import csrf_exempt
from django.conf import settings

from .runner import Runner
from .utils.urls import urls_parser
from .utils.workers import workers_manager
from .utils.serializer import MagicLinksSerializer

logger = logging.getLogger(__name__)


class BaseView(View):
    @method_decorator(csrf_exempt)
    def dispatch(self, request, *args, **kwargs):
        self.start_time = time.perf_counter()
        with request_context(request, endpoint=self), request_profiler(request, threshold=500):
            return super().dispatch(request, *args, **kwargs)

    def get_debug_info(self):
        execution_time = time.perf_counter() - self.start_time
        return {
            'duration': execution_time,
            'view': '{}.{}'.format(self.__class__.__module__, self.__class__.__name__),
            'method': self.request.method,
        }


class LinksView(BaseView):
    def post(self, request):
        urls = json.loads(request.body.decode())['links']
        runner = Runner(request, workers_manager, urls_parser)
        loop = asyncio.get_event_loop()
        result, *_ = loop.run_until_complete(asyncio.gather(runner.run(urls)))
        response_data = self.get_response_data(result)
        return JsonResponse(response_data, encoder=MagicLinksSerializer)

    def get_response_data(self, result):
        response_data = {'debug': self.get_debug_info(),
                         'drop_client_cache': settings.DROP_CACHE,
                         **result.to_dict()}
        return response_data
