import logging
import json
import itertools
import inspect
import urllib.parse
import asyncio
import aiohttp
import functools
import os
import re

import yenv
from abc import ABCMeta, abstractmethod
from copy import deepcopy
from ids.exceptions import BackendError
from ids.exceptions import IDSException
from django.conf import settings
from django.utils.functional import cached_property
from ylog.context import log_context
from concurrent import futures

from ..dto.result import Result
from ..utils.response import FailedResponse, is_failed_response
from ..dto.types import List, String, Image

logger = logging.getLogger(__name__)


class WorkerInterface(metaclass=ABCMeta):
    """Common interface for worker classes"""

    async def get(self):
        """
        Wrapper around get_result() method, catches all exceptions
        :rtype: Result
        """
        try:
            return await self.get_result()
        except Exception as exception:
            logger.exception('MAGICLINKS: Unhandled exception in worker %s: %s',
                             self.__class__.__name__, exception)
            return Result(data={})

    @abstractmethod
    async def get_result(self):
        """
        This is the public interface of each worker
        External consumers should invoke only it
        :rtype: Result
        """


class BaseWorker(WorkerInterface, metaclass=ABCMeta):
    HOSTNAME_REGEX = {'default': ''}
    PATH_REGEX = ''
    FRAGMENT_REGEX = ''
    QUERY_REGEX = ''
    FAVICON_TEXT = 'default'
    TTL_MAP = {'default': 200}

    default_timeout = 0.45

    def __init__(self, request, executor, session):
        """
        Request is an instance of current request, it contains user cookies and token
        which are used to authenticate in internal services
        :type request: WSGIRequest
        :type executor futures.ThreadPoolExecutor
        """
        self.request = request
        self.executor = executor
        self.url_objects = set()
        self.session = session

    def add_url_object(self, url_object):
        self.url_objects.add(url_object)

    def get_timeout(self):
        return self.default_timeout

    async def get_token(self):
        return settings.ROBOT_OAUTH_TOKEN

    def get_request_params(self):
        params = {
            'timeout': self.get_timeout()
        }
        return params

    def get_ttl(self, ttl_map=None, key='default'):
        if self.clear_cache:
            return -1
        ttl_map = ttl_map if ttl_map else self.TTL_MAP
        return ttl_map.get(key, ttl_map['default'])

    @cached_property
    def clear_cache(self):
        worker_file_name = os.path.basename(self.PATH).split('.')[0]
        return worker_file_name in settings.CLEAR_CACHE

    @cached_property
    def api_url(self):
        host_regex = re.match('\^(?P<host>.+)\$', self.HOSTNAME_REGEX[yenv.type])
        return host_regex.group('host')

    def handle_failed_response(self, url_object, response, completed=False):
        value = self.get_failed_value(url_object)
        return self.get_failed_result(value=value, completed=completed)

    def get_failed_value(self, url_object):
        return url_object.url

    def get_failed_result(self, value, completed=False):
        return List(
            ttl=self.get_ttl(key='fail'),
            value=[
                Image(
                    src=self.FAVICON,
                    text=self.FAVICON_TEXT,
                ),
                String(value=value),
            ],
            completed=completed,
        )

    async def has_access(self):
        return True

    def get_user_agent(self):
        return 'Hogwarts service'

    def __repr__(self):
        return inspect.getfile(self.__class__)


class HttpMixin(BaseWorker, metaclass=ABCMeta):
    method = 'GET'
    response_attr = 'json'

    async def fetch(self, request, url_object):
        """
        Used to catch Request exceptions and log them
        :type request: dict
        :type url_object: dto.UrlObject
        :rtype: json|FailedResponse
        """
        content_type_filter = self.get_content_type_filter()
        try:
            if content_type_filter:
                head_request = deepcopy(request)
                head_request['method'] = 'HEAD'
                async with self.session.request(**head_request) as response:
                    response.raise_for_status()
                    headers = response.headers
                    content_type = headers.get('Content-Type', '').split(';')[0]
                    content_type = content_type.strip().lower()
                    if content_type not in content_type_filter:
                        return '', url_object

            async with self.session.request(**request) as response:
                result = await getattr(response, self.response_attr)()
                response.raise_for_status()
        except (BackendError,
                aiohttp.ClientResponseError,
                aiohttp.client_exceptions.ClientConnectorError,
                asyncio.TimeoutError,
                futures._base.TimeoutError,
                ) as error:

            if isinstance(error, aiohttp.ClientResponseError) and error.status == 404:
                pass
            else:
                with log_context(request_data=request.get('data') or request.get('json')):
                    logger.exception('Fetching %s failed', request['url'])

            result = FailedResponse(request, error)
        return result, url_object

    def get_request_method(self):
        return self.method

    def check_hidereferer(self, url_object):
        url = url_object.url
        if url_object.hide_ref:
            return self.hideref_url(url)
        return url

    def hideref_url(self, url):
        url = urllib.parse.quote(url, safe='')
        return 'https://{}/?{}'.format(settings.HIDE_REF_NETLOC, url)

    async def get_auth_header(self):
        token = await self.get_token()
        return 'OAuth {}'.format(token)

    async def get_headers(self):
        oauth_header = await self.get_auth_header()
        return {'Authorization': oauth_header}

    def get_content_type_filter(self):
        return None


class BulkMixin(metaclass=ABCMeta):
    def handle_response(self, response):
        if is_failed_response(response):
            return self.handle_failed_response(response)
        return self.handle_successful_response(response)

    def handle_failed_response(self, response, completed=False):
        data = {}
        for url_object in self.url_objects:
            value = self.get_failed_value(url_object)
            data[url_object.url] = self.get_failed_result(value=value, completed=completed)
        return Result(data=data, completed=False)

    @abstractmethod
    def handle_successful_response(self, response):
        """
        :type response: Response
        :rtype: Result
        """


class HttpBaseWorker(HttpMixin, metaclass=ABCMeta):
    async def get_result(self):
        """
        Retrieving information about given url_objects
        :rtype: Result
        """
        result_obj = Result(data={})
        has_access = await self.has_access()
        if not has_access:
            for url_object in self.url_objects:
                result_obj.data[self.check_hidereferer(url_object)] = self.handle_failed_response(
                    url_object, FailedResponse())
        else:
            tasks = [
                self.fetch(request=request, url_object=url_object)
                for request, url_object in await self.create_requests()
            ]
            for future in asyncio.as_completed(tasks):
                response, url_object = await future
                result_obj.data[self.check_hidereferer(url_object)] = self.handle_response(url_object, response)

        return result_obj

    @abstractmethod
    async def create_requests(self):
        """
        :return: tuple(tuple(requests.Request, url_object.UrlObject))
        """

    def create_request(self, url_object, headers, api_name=None, params=None, dumps=False):
        """
        :type url_object: Link
        :type headers: dict
        :type api_name: basestring
        :type params: dict
        :type dumps: bool
        :rtype: dict
        """
        request_data = self.get_request_data(url_object, api_name, params)
        url = self.get_request_url(url_object, api_name)
        method = self.get_request_method()
        request_json = self.get_request_json(url_object, api_name, params)

        return dict(
            url=url,
            method=method,
            data=json.dumps(request_data) if dumps else request_data,
            json=request_json,
            headers=headers,
            **self.get_request_params()

        )

    @abstractmethod
    def get_request_url(self, url_object, api_name):
        """
        Returns request link for given link
        :type url_object: Link
        :type api_name: basestring
        :rtype: str
        """

    def get_request_data(self, url_object, api_name, params):
        """
        Can be overridden to provide POST data (key-value pairs)
        :type url_object: Link
        :type api_name: basestring
        :type params: dict
        :rtype: dict
        """

    def get_request_json(self, url_object, api_name, params):
        """
        Can be overridden to provide JSON payload as request body
        :type url_object: Link
        :type api_name: basestring
        :type params: dict
        :rtype: dict
        """

    @abstractmethod
    def handle_response(self, url_object, response):
        """
        :type url_object: Link
        :type response: HttpResponse|FailedResponse
        :rtype: Type
        """

    @abstractmethod
    def handle_successful_response(self, url_object, response):
        pass


class HttpSingleWorker(HttpBaseWorker, metaclass=ABCMeta):

    async def create_requests(self):
        headers = await self.get_headers()
        return ((self.create_request(url_object=url_object,
                                     headers=headers,
                                     ), url_object)
                for url_object in self.url_objects)

    def handle_response(self, url_object, response):
        """
        :type url_object: Link
        :type response: HttpResponse|FailedResponse
        :rtype: Type
        """
        if is_failed_response(response):
            return self.handle_failed_response(url_object, response)
        return self.handle_successful_response(url_object, response)


class HttpMultiWorker(HttpBaseWorker, metaclass=ABCMeta):
    """
    Этот базовый воркер не был доработан для поддержки ассинхронности в связи
    с отсутствием в момент разработки воркеров, которые бы его использовали,
    если вновь есть потребность его использования - следует изменить его код
    """

    def fetch(self, tuple_of_requests):
        """
        Used when we need to make couple of requests for each link
        :type tuple_of_requests: tuple(requests.Request)
        :rtype: tuple(requests.Response|FailedResponse)
        """
        pool = self.executor()
        responses = pool.map(super().fetch, tuple_of_requests)
        return responses

    def create_requests(self):
        """
        :rtype: tuple(tuple(requests.Request))
        """
        return (self.create_requests_for_object(url_object) for url_object in self.url_objects)

    @abstractmethod
    def create_requests_for_object(self, url_object):
        """
        :type url_object: Link
        :rtype: tuple(requests.Request)
        """

    @abstractmethod
    def merge_responses(self, responses):
        """
        :type responses: tuple(requests.Response)
        :rtype: dict
        """

    def handle_response(self, url_object, responses):
        """
        :type url_object: Link
        :type responses: tuple(HttpResponse|FailedResponse)
        :rtype: Type
        """
        responses, _false_check = itertools.tee(responses)
        if any(is_failed_response(copy_response) for copy_response in _false_check):
            return self.handle_failed_response(url_object, responses)

        return self.handle_successful_response(url_object, responses)


class HttpBulkWorker(BulkMixin, HttpMixin, metaclass=ABCMeta):
    async def get_result(self):
        response = await self.get_response()
        return self.handle_response(response)

    async def get_response(self):
        headers = await self.get_headers()
        request = self.create_request(headers=headers)
        task = self.fetch(request=request,
                          url_object=None,
                          )
        response, _ = await task
        return response

    def create_request(self, headers):
        """
        :type headers dict
        :rtype: dict
        """
        return dict(
            url=self.get_request_url(),
            method=self.get_request_method(),
            data=self.get_request_data(),
            json=self.get_request_json(),
            headers=headers,
            **self.get_request_params()
        )

    @abstractmethod
    def get_request_url(self):
        """
        :rtype: str
        """

    def get_request_data(self):
        """
        :rtype: dict
        """

    def get_request_json(self):
        """
        :rtype: dict
        """


class IdsMixin(metaclass=ABCMeta):
    @abstractmethod
    async def get_repository(self):
        """
        Returns IDS repository
        :rtype: Repository
        """

    @abstractmethod
    def get_lookup_data(self):
        """
        Returns the lookup data to use with repository
        :rtype: dict
        """


class IdsBulkWorker(IdsMixin, BulkMixin, BaseWorker, metaclass=ABCMeta):
    async def get_result(self):
        repository = await self.get_repository()
        lookup_data = self.get_lookup_data()
        loop = asyncio.get_event_loop()
        get_data_from_repository = functools.partial(
            repository.get,
            lookup_data
        )
        task = loop.run_in_executor(self.executor, get_data_from_repository)

        try:
            response = await task
        except IDSException as error:
            logger.exception('MAGICLINKS: IDS Exception occurred. Repository: %s. '
                             'Lookup data: %s', repository, lookup_data)
            response = FailedResponse(error=error)
        return self.handle_response(response)
