# coding: utf-8
import asyncio
import functools
import re

from urllib.parse import unquote

from django.conf import settings

import blackbox

from intranet.magiclinks.src.links.utils.tvm_client import get_service_ticket


class BusinessMixin(object):
    """
    Объединяет методы, которые
    используются в магических
    ссылках для бизнеса
    """

    async def fetch_data(self, request):
        async with self.session.request(**request) as response:
            result = await response.json()
            response.raise_for_status()
        return result

    async def parse_ycookie(self, ycookie):
        if not ycookie or not isinstance(ycookie, str):
            return {}

        if re.match(r'%23\d{10}', ycookie):
            # fix of encoded  # => %23 in YP-cookie
            ycookie = unquote(ycookie)

        ycookie = ycookie.split('#')

        result = {}
        for cookie in ycookie:
            items = cookie.split('.')
            if len(items) == 2:
                result[items[0]] = unquote(items[1])
            elif len(items) > 2:
                try:
                    int(items[0])
                    result[items[1]] = unquote('.'.join(items[2:]))
                except ValueError:
                    result[items[0]] = unquote('.'.join(items[1:]))

        return result

    async def get_user_org_id(self):
        if hasattr(self.request, 'org_id'):
            return self.request.org_id

        yp = await self.parse_ycookie(self.request.COOKIES.get('yp', ''))
        org_id_from_request = yp.get('org_id')

        blackbox_info = await self.get_user_data(self.request.yauser.uid)
        users_data = blackbox_info.get('users', [])
        if users_data:
            orgs_from_bb = users_data[0].get('attributes', {}).get('1017')
        else:
            raise ValueError('Got no users from blackbox "{}"'.format(blackbox_info))

        if orgs_from_bb:
            orgs_from_bb = orgs_from_bb.split(',')
        else:
            raise ValueError('User with uid={} does not have an organization'.format(self.request.yauser.uid))

        if org_id_from_request in orgs_from_bb:
            return org_id_from_request
        else:
            return orgs_from_bb[0]

    def create_request_url(self, host, version, method):
        return '{}/{}/{}'.format(host, version, method)

    async def get_user_data(self, uid_data):
        """
        Получаем информацию о пользователях из blackbox по uid

        :type uid_data: dict/str
        """
        ticket_for_blackbox = await get_service_ticket(settings.BLACKBOX_CLIENT.value, self.executor)

        black_box_caller = blackbox.JsonBlackbox(url=blackbox.BLACKBOX_URL)
        loop = asyncio.get_event_loop()
        get_userinfo = functools.partial(black_box_caller.userinfo,
                                         userip=self.request.user_ip,
                                         uid=uid_data,
                                         headers={'X-Ya-Service-Ticket': ticket_for_blackbox},
                                         attributes=1017,
                                         )
        task = loop.run_in_executor(self.executor, get_userinfo)
        user_data = await task
        return user_data


class StartrekForBusinessMixin(BusinessMixin):

    async def get_headers(self):
        headers = await super().get_headers()
        if not self.demo:
            user_org_id = await self.get_user_org_id()
            headers.update({'x-org-id': user_org_id})
        return headers


class StaffForBusinessMixin(BusinessMixin):

    async def get_headers(self):
        headers = await super().get_headers()
        user_org_id = await self.get_user_org_id()
        tvm2_service_ticket = await get_service_ticket(self.TVM2_DESTINATION_CLIENT_ID, self.executor)
        headers.update({'x-org-id': user_org_id, 'X-Ya-Service-Ticket': tvm2_service_ticket})
        return headers
