import logging
from concurrent.futures import ThreadPoolExecutor

from django.conf import settings
from django.core.handlers.wsgi import WSGIRequest

from ...utils.tvm_client import get_service_ticket
from .base import BusinessMixin
from ..intranet.inflector import Worker as InflectorIntranetWorker, UserNotExists

logger = logging.getLogger(__name__)


class Worker(InflectorIntranetWorker):
    PATH_REGEX = r'^/(?P<login>[\w\-.]+)/?$'  # начинается '/', содержит [\w, '-', '.'] и может заканчиваться '/'
    TVM2_DESTINATION_CLIENT_ID = settings.DIR_TVM2_CLIENT_ID

    async def request_staff(self, login: str) -> dict[str, ...]:
        staff = await self.make_request_to_directory(login=login)
        return self.staff_like_intranet(staff)

    async def make_request_to_directory(self, login: str) -> dict[str, ...]:
        url = f'{settings.DIR_HOST}/v10/users'
        headers = await self.get_headers()
        params = {'nickname': login, 'fields': 'name,gender,is_dismissed'}

        async with self.session.request('get', url=url, headers=headers, params=params) as response:
            result = await response.json()
            response.raise_for_status()

        users = result['result']

        if not users:
            logger.exception(f'MAGICLINKS: Not found user in directory by login [{login}]')
            raise UserNotExists

        return users[0]

    async def get_headers(self) -> dict[str, str]:
        user_org_id = await UserOrgId(self.request, self.executor).get_user_org_id()
        tvm2_service_ticket = await get_service_ticket(self.TVM2_DESTINATION_CLIENT_ID, self.executor)
        return {'x-org-id': user_org_id, 'X-Ya-Service-Ticket': tvm2_service_ticket}

    @staticmethod
    def staff_like_intranet(staff: dict[str, str | dict[str, str]]) -> dict[str, ...]:
        first_name, last_name = staff['name']['first'], staff['name']['last']
        gender = staff['gender']  # ['male', 'female', 'unspecified']
        is_dismissed = staff['is_dismissed']

        return {
            'personal': {'gender': gender},
            'memorial': None,
            'official': {'is_dismissed': is_dismissed},
            'name': {
                'first': {'ru': first_name, 'en': first_name},
                'last': {'ru': last_name, 'en': last_name},
            }
        }


class UserOrgId(BusinessMixin):
    """only for user function `get_user_org_id`"""
    def __init__(self, request: WSGIRequest, executor: ThreadPoolExecutor):
        self.request = request
        self.executor = executor
