import logging
import collections

from django.conf import settings

from .base import StartrekForBusinessMixin
from ..intranet.startrek_issue import Worker as IntranetWorker
from ..mixins.async_response import AsyncResponseMixin
from ...dto import Result, String

logger = logging.getLogger(__name__)


class Worker(StartrekForBusinessMixin, AsyncResponseMixin, IntranetWorker):
    HOSTNAME_REGEX = {'production': '^(demo\.)?tracker.yandex.(ru|com)$',
                      'preprod': '^tracker.preprod.yandex.(ru|com)$',
                      'testing': '^(demo\.)?tracker.test.yandex.(ru|com)$',
                      'development': '^(demo\.)?tracker.test.yandex.(ru|com)$',
                      }
    PATH_REGEX = '^/(?P<key>[A-Z]+-\d+)/?$'

    @property
    def demo(self):
        url_object = next(iter(self.url_objects))
        return 'demo' in url_object.split_result.netloc

    @property
    def api_url(self):
        if self.demo:
            return settings.STARTREK_DEMO_API_URL
        return settings.STARTREK_API_URL

    async def get_prepared_data(self, response):
        """
        Так как внешнее апи трекера отдает
        uid вместо логина - заменяем uid на
        логин используя информацию из блекбокса
        """
        tickets_map = super().get_prepared_data(response)
        assignee_tickets = self.map_uids(tickets_map)
        if not assignee_tickets:
            return tickets_map
        assignee_blackbox_info = await self.get_user_data(assignee_tickets)
        return self.add_logins_to_response(tickets_map, assignee_tickets, assignee_blackbox_info)

    def map_uids(self, tickets_map):
        """
        Возвращает словарь в котором элементы
        формируются следующим образом:
        uid - > set(номера тикетов, в которых указан данный uid)
        :type tickets_map: dict
        :rtype: dict
        """
        uids_data = collections.defaultdict(set)

        for key, result in tickets_map.items():
            assignee = result.get('assignee')
            if assignee:
                uids_data[result['assignee']['id']].add(key)

        return uids_data

    def add_logins_to_response(self, tickets_map, assignee_tickets, assignee_blackbox_info):
        """
        Проходим по ответу блекбокса, если
        для uid не получили в ответ error и получили
        login (значит такой пользователь есть) -
        проходим по всем тикетам, где встречается
        указанный uid и заменяем uid на логин
        полученный в ответе от блекбокса (так как
        блекбокс отдает логин в виде test@connect.yandex.ru
        берем только часть до @)
        :type tickets_map: dict
        :type assignee_tickets: dict
        :type assignee_blackbox_info: dict
        :rtype: dict
        """
        users_data = assignee_blackbox_info.get('users', [])
        if not users_data:
            logger.warning('Got no users from blackbox "%s"', assignee_blackbox_info)
        for user_data in users_data:
            user_uid = user_data['id']
            error = user_data.get('error')
            if error:
                logger.error('Got error "%s" while getting login with "%s"',
                             error, user_uid)
                continue

            user_login = user_data.get('login')
            if not user_login:
                logger.info('User with uid "%s" doesnt exists', user_uid)
                continue

            for ticket_key in assignee_tickets[user_uid]:
                tickets_map[ticket_key]['assignee']['id'] = user_login.split('@')[0]

        return tickets_map

    def add_assignee(self, data, url_object, url_data):
        if not self.demo:
            super().add_assignee(data, url_object, url_data)
        else:
            assignee_data = url_data.get('assignee')
            if assignee_data is not None:
                data[url_object.url].value.append(String(value='- {}'.format(assignee_data['id'])))

    async def handle_successful_response(self, response):
        data = {}
        response_data = await self.get_prepared_data(response)
        for url_object in self.url_objects:
            self.add_result_for_object(response_data=response_data,
                                       url_object=url_object,
                                       data=data,
                                       )

        return Result(data=data)
