import re
import collections
import os
import logging

from django.conf import settings
from bs4 import BeautifulSoup

from intranet.magiclinks.src.links.utils.tvm_client import get_service_ticket_sync
from ..base import HttpSingleWorker
from ...dto.types import List, String, Image

logger = logging.getLogger(__name__)


class Worker(HttpSingleWorker):
    HOSTNAME_REGEX = {'default': '^(?!.*yandex-team).+',
                      }
    PATH_REGEX = r'.*'

    OPEN_GRAPH_PROPERTIES = collections.OrderedDict(
        (
            ('title', 'get_title'),
            ('site_name', 'get_site_name'),
         )
    )

    PATH = os.path.abspath(__file__)

    OPEN_GRAPH_META_REGEX = re.compile(r'^og:({})$'.format('|'.join(OPEN_GRAPH_PROPERTIES)))

    response_attr = 'text'

    TTL_MAP = {'default': 86400,  # 1 день
               'fail': 1200,  # 20 минут
               }

    def get_timeout(self):
        return 2

    def get_request_url(self, url_object, api_name=None):
        url = url_object.url
        logger.info(f'Open graph url: {url}')
        return url

    def get_opengraph_data(self, response):
        result = dict()
        parsed_response = BeautifulSoup(response, "html.parser")
        head = parsed_response.head
        if head is not None:
            open_graph_tags = head.find_all(name='meta',
                                            property=self.OPEN_GRAPH_META_REGEX,
                                            )
            for tag in open_graph_tags:
                if tag.has_attr('content'):
                    # [3:] так как заголовки имеют вид og_title
                    result[tag['property'][3:]] = tag['content']

        return result

    def get_favicon(self, url_object):
        host = url_object.split_result.netloc
        return settings.FAVICON_SITE_TEMPLATE.format(host)

    def get_site_name(self, site_name):
        return '@{}'.format(site_name)

    def get_title(self, title):
        if len(title) > 30:
            title = '{}... '.format(title[:26])
        return title

    def get_result_data(self, opengraph_data):
        data = [
            getattr(self, func)(opengraph_data[og_title])
            for og_title, func in self.OPEN_GRAPH_PROPERTIES.items()
            if opengraph_data.get(og_title) is not None
        ]
        return data

    def make_result(self, url_object, ttl, data=None):
        favicon = self.get_favicon(url_object)
        if data is None or len(data) == 0:
            data = [url_object.url, ]

        result = List(
            ttl=ttl,
            value=[
                List(ttl=ttl,
                     value=[
                         Image(
                             src=favicon,
                             text='OpenGraph',
                         ),
                     ],
                     action={
                         "event": "click",
                         "type": "halfscreenpreview",
                         "url": url_object.url,
                     }),
            ]
        )
        for value in data:
            result.value.append(String(value=value))
        return result

    def handle_successful_response(self, url_object, response):
        logger.info(f'Open graph successful response, url: {url_object.url}')
        opengraph_data = self.get_opengraph_data(response)
        result_data = self.get_result_data(opengraph_data)
        logger.info(f'Open graph successful response, result_data: {result_data}')
        return self.make_result(url_object=url_object,
                                ttl=self.get_ttl(),
                                data=result_data,
                                )

    def handle_failed_response(self, url_object, response):
        logger.info(f'Open graph failed response, url: {url_object.url}')
        return self.make_result(url_object=url_object,
                                ttl=self.get_ttl(key='fail'),
                                )

    async def get_headers(self):
        headers = await super().get_headers()
        zora_headers = {
            'X-Ya-Follow-Redirects': 'True',
            'X-Ya-Ignore-Certs': 'True',
        }
        headers.update(zora_headers)
        return headers

    def get_request_params(self):
        tvm = get_service_ticket_sync(settings.GOZORA_TVM2_CLIENT_ID)
        user = settings.GOZORA_SOURCE_NAME
        proxy = f'http://{user}:{tvm}@go.zora.yandex.net:1080/'
        params = super().get_request_params()
        params['proxy'] = proxy
        return params

    def get_content_type_filter(self):
        return ['text/html']
