import os
from django.conf import settings

from ..base import HttpSingleWorker
from ..mixins.check_internal_access import CheckInternalAccessMixin
from ...dto.types import List, String, Image, User, BinaryResource


class Worker(CheckInternalAccessMixin, HttpSingleWorker):
    HOSTNAME_REGEX = {'production': '^abc.yandex-team.ru$',
                      'testing': '^abc.test.yandex-team.ru$',
                      'development': '^abc.yandex-team.ru$',
                      }
    PATH_REGEX = '/services/(?P<service>[-\w]+)/?$'

    PATH = os.path.abspath(__file__)

    PLAY = BinaryResource('play.ico', PATH)
    PAUSE = BinaryResource('pause.ico', PATH)
    STOP = BinaryResource('stop.ico', PATH)
    QUESTION = BinaryResource('question.ico', PATH)
    FAVICON = BinaryResource('favicon.ico', PATH)
    FAVICON_TEXT = 'ABC'

    TTL_MAP = {
        'default': 86400,  # 1 день
        'fail': 60,
        'develop': 86400,
        'closed': 604800,  # 1 неделя
        'deleted': 604800,
        'frozen': 604800,
        'supported': 604800,
        'needinfo': 604800,
    }

    favicon_map = {
        'closed': STOP,
        'deleted': STOP,
        'develop': PLAY,
        'frozen': PAUSE,
        'supported': PAUSE,
        'needinfo': QUESTION,
    }

    FIELDS = ('id', 'owner.login', 'state', 'name.ru', 'unique_immediate_members_count',)

    def get_request_url(self, url_object, api_name=None):
        service = url_object.path_match.group('service')

        return 'https://{}/api/v4/services/?slug={}&fields={}'.format(settings.ABC_API_URL,
                                                                      service,
                                                                      ','.join(self.FIELDS),
                                                                      )

    def format_persons(self, persons_in_team):
        if persons_in_team == 0:
            return '(Нет участников)'
        format_members = self.get_members_declension(persons_in_team)

        return '({} {})'.format(persons_in_team, format_members)

    def get_service_members(self, response):
        return int(response.get('unique_immediate_members_count', 0))

    def get_members_declension(self, persons_in_team):
        choices = ['участник', 'участника', 'участников']
        inumber = persons_in_team % 100
        if 11 <= inumber <= 19:
            choice = choices[2]
        else:
            inumber = inumber % 10
            if inumber == 1:
                choice = choices[0]
            elif inumber in [2, 3, 4]:
                choice = choices[1]
            else:
                choice = choices[2]
        return choice

    def handle_successful_response(self, url_object, response):
        response = response['results']
        if not response:
            return self.handle_failed_response(url_object, response)

        response = response[0]
        status = response['state']
        state_favicon = self.favicon_map.get(status, '')
        service_name = response['name']['ru']

        persons_in_team = self.get_service_members(response)

        result = List(
            ttl=self.get_ttl(key=status),
            value=[
                Image(
                    src=state_favicon,
                    text=self.FAVICON_TEXT,
                ),
                String(value=service_name),
            ]
        )

        owner = response.get('owner')
        if owner:
            login = owner.get('login')
            if login:
                result.value.append(User(login=login))

        result.value.append(String(value=self.format_persons(persons_in_team), color='#999999'))

        return result
