import datetime
import os

from django.conf import settings

from ..base import HttpSingleWorker
from ...dto.types import List, String, Image, User, BinaryResource


class Worker(HttpSingleWorker):
    HOSTNAME_REGEX = {'production': '^abc.yandex-team.ru$',
                      'testing': '^abc.test.yandex-team.ru$',
                      'development': '^abc.yandex-team.ru$',
                      }
    PATH_REGEX = '/services/(?P<service>[-\w]+)/duty/?$'
    QUERY_REGEX = 'role=(?P<role_id>\d+)/?'

    PATH = os.path.abspath(__file__)

    FAVICON = BinaryResource('favicon.ico', PATH)
    FAVICON_TEXT = 'ABC Duty'

    TTL_MAP = {
        'default': 1800,  # 30 минут
        'fail': 60,
    }

    MONTH_MAP = {
        1: 'января',
        2: 'февраля',
        3: 'марта',
        4: 'апреля',
        5: 'мая',
        6: 'июня',
        7: 'июля',
        8: 'августа',
        9: 'сентября',
        10: 'октября',
        11: 'ноября',
        12: 'декабря',
    }

    def get_request_url(self, url_object, api_name=None):
        role_id = url_object.query_match.group('role_id')

        return 'https://{}/api/v4/duty/on_duty/?schedule={}'.format(
            settings.ABC_API_URL,
            role_id,
        )

    def format_date(self, response):
        date_parsed = datetime.datetime.strptime(response['end'], '%Y-%m-%d')
        return '(до {}-го {})'.format(date_parsed.day, self.MONTH_MAP[date_parsed.month])

    def collect_duty(self, response):
        result = []
        for duty_person in response:
            duty_login = (duty_person.get('person') or {}).get('login')
            if duty_login:
                result.append(User(login=duty_login))
        if result:
            if len(result) == 1:
                result = result[0]
            else:
                result = List(ttl=self.get_ttl(), value=result, separator=', ')
        return result

    def handle_successful_response(self, url_object, response):
        if not len(response):
            return self.handle_failed_response(url_object, response, completed=True)

        response_core = response[0]
        schedule_name = response_core['schedule']['name']
        end_date =  self.format_date(response_core)

        result = List(
            ttl=self.get_ttl(),
            value=[
                Image(
                    src=self.FAVICON,
                    text=self.FAVICON_TEXT,
                ),
                String(value=schedule_name),
            ]
        )
        collected_duty = self.collect_duty(response)
        if collected_duty:
            result.value.append(collected_duty)
        else:
            result.value.append(String(value='Не назначен', color='red'))

        result.value.append(String(value=end_date, color='#999999'))
        return result
