# coding: utf-8
import logging
import json
import os
import pytz

from dateutil.parser import parse as parse_dt
from urllib.parse import urlunparse

from django.conf import settings

from ..base import HttpSingleWorker
from ...dto.types import List, String, Image, BinaryResource


logger = logging.getLogger(__name__)


MSK_TZ = pytz.timezone('Europe/Moscow')


class Worker(HttpSingleWorker):
    HOSTNAME_REGEX = {
        'production': '^(cab|eds).yandex-team.ru$',
        'testing': '^(cab|eds).test.tools.yandex-team.ru$',
        'development': '^(cab|eds).test.tools.yandex-team.ru$',
    }
    PATH_REGEX = '^/(?:eds/)?(document|file)/(?P<id>\d+)/?$'

    SIGNED_TO_STATUS_DATA = {
        False: ('ожидает подписания {sign_type}', '#eabf3d'),
        True: ('подписан', '#3dbf41'),
    }

    PATH = os.path.abspath(__file__)

    method = 'POST'

    FAVICON = BinaryResource('favicon.ico', PATH)
    FAVICON_TEXT = 'EDS'

    TTL_MAP = {
        'default': 60 * 10,
        'fail': 60,
    }

    async def get_token(self):
        return settings.EDS_TOKEN

    async def get_headers(self):
        headers = await super().get_headers()
        headers.update({'Content-Type': 'application/json'})
        return headers

    def get_timeout(self):
        return 5

    @property
    def api_url(self):
        return settings.EDS_API_URL

    def get_request_url(self, url_object, api_name=None):
        return 'https://{}/rest/getDssDocInfo'.format(self.api_url)

    def get_request_data(self, url_object, api_name=None, params=None):
        return json.dumps({
            'filters': {
                'docId': url_object.path_match.group('id')
            }
        })

    def get_404_result(self):
        value = [
            Image(
                src=self.FAVICON,
                text=self.FAVICON_TEXT,
            ),
            String(
                value='Документ не найден',
                color='#999999',
            ),
        ]

        return List(
            ttl=self.TTL_MAP['default'],
            value=value
        )

    def handle_successful_response(self, url_object, response):
        process = response['proc']
        if process:
            docs = process[0]['doc']
        else:
            # No such document
            return self.get_404_result()

        if docs:
            doc = docs[0]
        else:
            logger.error('No documents for "%s"', url_object.url)
            return self.get_404_result()

        signed = doc['status'] == 'SIGNED'
        canceled = doc['status'] == 'CANCELED'
        name = doc['docNameRu']
        is_eds = doc['signType'] == 'DSS'
        deadline_expired = doc['deadlineExpired']

        if canceled:
            status_label = 'отменен'
            status_color = '#999999'
        else:
            status_label, status_color = self.SIGNED_TO_STATUS_DATA[signed]
            status_label = status_label.format(sign_type='ЭП' if is_eds else 'вручную')

        # Preview is always points to /eds/file/<doc-id>/?_embedded=1
        scheme, netloc, path, query, fragment = url_object.split_result
        path = path.replace('eds/', '', 1)
        path = path.replace('document', 'file', 1)
        netloc = netloc.replace('cab', 'eds')

        if query:
            query = '{}&_embedded=1'.format(query)
        else:
            query = '_embedded=1'
        url = urlunparse((scheme, netloc, path, None, query, fragment))

        action = {
            'event': 'click',
            'type': 'halfscreenpreview',
            'url': url,
        }

        value = [
            Image(
                src=self.FAVICON,
                text=self.FAVICON_TEXT,
            ),
            String(
                value='[{}]'.format(status_label),
                color=status_color,
                action=action,
            ),
            String(
                value=name,
                action=action,
            ),
        ]

        deadline = doc.get('deadline')  # format is '2018-11-11T12:51:50Z'
        if not signed and not deadline_expired and is_eds and deadline is not None:
            deadline = parse_dt(deadline).replace(tzinfo=pytz.UTC).astimezone(tz=MSK_TZ).strftime("%d.%m.%Y")
            deadline_label = 'истекает {deadline}'.format(deadline=deadline)

            value.append(
                String(
                    value=deadline_label,
                    color='#999999',
                    action=action,
                )
            )

        return List(
            ttl=self.TTL_MAP['default'],
            value=value
        )
