from ..base import HttpSingleWorker
from ..mixins.femida import FemidaMixin
from ..mixins.person_ticket import PersonTicketMixin
from ...dto import List, Image, String, User


class Worker(FemidaMixin, PersonTicketMixin, HttpSingleWorker):

    PATH_REGEX = r'^/(?P<resource>(interviews|vacancies|problems))/(?P<id>\d+)/?$'

    VACANCY_STATES = {
        'closed': ('Закрыта', '#425EC3'),
        'draft': ('Черновик', '#000000'),
        'in_progress': ('В работе', '#049657'),
        'offer_accepted': ('Оффер принят', '#6EC5EF'),
        'offer_processing': ('Делаем оффер', '#F7A014'),
        'on_approval': ('На согласовании', '#F7A014'),
        'suspended': ('Приостановлена', '#000000'),
    }

    STATE_COLOR = {
        'Завершена': '#425EC3',
        'Назначена': '#F7A014',
        'В работе': '#049657',
        'Оценена': '#6EC5EF',
        'В архиве': '#000000',
        'Отменена': '#000000',
    }

    DEFAULT_COLOR = '#888C91'

    QUERY_FIELDS = {
        'interviews': ('section', 'state_verbose', 'candidate_name', 'grade'),
        'vacancies': ('name', 'hiring_manager', 'status'),
        'problems': ('summary', 'created_by', 'stats'),
    }

    TTL_MAP_INTERVIEW = {
        'default': 1800,
        'Завершена': 172800,  # 2 дня
        'Оценена': 172800,
        'В архиве': 172800,
        'Отменена': 172800,
    }

    TTL_MAP_VACANCY = {
        'default': 1800,
        'Закрыта': 604800,
        'Приостановлена': 86400,
        'Черновик': 86400,
    }

    def get_request_url(self, url_object, api_name=None):
        resource = url_object.path_match.group('resource')
        return 'https://{}/api/{}/{}/?_fields={}'.format(url_object.split_result.netloc,
                                                         resource,
                                                         url_object.path_match.group('id'),
                                                         self.get_fields_for_resource(resource))

    def get_fields_for_resource(self, resource):
        return ','.join(self.QUERY_FIELDS[resource])

    def handle_successful_response(self, url_object, response):
        resource = url_object.path_match.group('resource')
        if resource == 'interviews':
            return self.handle_response_for_interview(response)
        elif resource == 'problems':
            return self.handle_response_for_problem(response)
        elif resource == 'vacancies':
            return self.handle_response_for_vacancy(response)

    def get_color_for_state(self, state):
        return self.STATE_COLOR.get(state, self.DEFAULT_COLOR)

    def get_state_for_vacancy(self, data):
        state = data['status']
        status, color = self.VACANCY_STATES.get(state, ('Not defined', '#888C91'))
        return status, color

    def handle_response_for_vacancy(self, data):
        vacancy_name = data['name']

        status, color = self.get_state_for_vacancy(data)
        result = List(
            ttl=self.get_ttl(ttl_map=self.TTL_MAP_VACANCY,
                             key=status),
            value=[
                self.favicon,
                String(value=status, color=color),
                String(value=vacancy_name),
            ]
        )
        hiring_manager = data.get('hiring_manager')
        if hiring_manager:
            manager_login = hiring_manager.get('login')
            if manager_login:
                result.value.append(User(login=manager_login))

        return result

    def handle_response_for_problem(self, data):
        title = data['summary']
        author = data['created_by']['login']

        result = List(
            ttl=self.get_ttl(key='problems'),
            value=[
                self.favicon,
                String(value=title),
                User(login=author),
            ]
        )
        stats = data.get('stats')
        if stats:
            result.value.extend((
                String(value='('),
                String(value=stats['bad'], color='#ff0000'),
                String(value=stats['norm'], color='#D49300'),
                String(value=stats['excellent'], color='#049657'),
                String(value=')'),
            )
            )
        return result

    def handle_response_for_interview(self, data):
        applicant_name = data['candidate_name']
        section = data['section']

        state = data['state_verbose']

        result = List(
            ttl=self.get_ttl(ttl_map=self.TTL_MAP_INTERVIEW,
                             key=state),
            value=[
                self.favicon,
                String(value=applicant_name),
                String(value=' - секция'),
                String(value=section),
                String(value='({})'.format(state), color=self.get_color_for_state(state)),
            ]
        )

        grade = data.get('grade')
        if grade is not None:
            result.value.append(
                Image(
                    src=self.interview_grade_icons.get(data['grade'], ''),
                    text=str(grade)
                ),
            )
        return result
