import datetime
import os
import pytz
from django.conf import settings
from django.utils import timezone

from .startrek_base import StartrekBase
from ...dto.types import List, String, Image, User, BinaryResource


class Worker(StartrekBase):
    HOSTNAME_REGEX = {'production': r'^goals.yandex-team.ru$',
                      'testing': r'^goals.test.tools.yandex-team.ru$',
                      'development': r'^goals.yandex-team.ru$',
                      }
    PATH_REGEX = r'^/(filter|okr|compilations\/\w+)$'
    QUERY_REGEX = r'.*goal=(?P<id>\d+).*'

    STATUS_DATA = {
        'newGoal': ('Новая', '#027bf3'),
        'draft': ('Черновик', '#027bf3'),

        'asPlanned': ('По плану', '#07a300'),
        'readyForPmc': ('Готово к PMC', '#07a300'),
        'approvedPmc': ('Утверждено PMC', '#07a300'),
        'approvedGrandPmc': ('Утверждено Grand PMC', '#07a300'),

        'withRisks': ('Есть риски', '#f5ab19'),
        'approvalFromTheVsHead': ('Согласование у руководителя VS', '#f5ab19'),

        'blockedGoal': ('Заблокирована', '#ff0400'),

        'cancelled': ('Отменена', '#888C91'),
        'achieved': ('Достигнута', '#888C91'),
        'partiallyCompleted': ('Частично достигнута', '#888C91'),
        'notDo': ('Не делаем', '#888C91'),
        'rejectedByPmc': ('Отклонено PMC', '#888C91'),
    }

    IMPORTANCE_DATA = {
        'default': ('Unknown', '⚽'),
        'team': ('Casual', ' '),
        'department': ('Star', '⭐'),
        'company': ('Crown', '👑'),
        'OKR': ('OKR', '🔻'),
    }

    PATH = os.path.abspath(__file__)

    FAVICON = BinaryResource('favicon.ico', PATH)
    FAVICON_TEXT = 'Goals'

    TTL_MAP = {
        'default': 300,  # 5 мин
        'fail': 60,
        'Отменена': 86400 * 30,  # 30 дней
        'Достигнута': 86400 * 30,
    }

    TICKET_FIELDS = 'assignee', 'summary', 'status', 'aliases', 'resolution', 'deadline', 'goalImportance'

    STARTREK_QUEUE = settings.STARTREK_QUEUE

    def get_key_from_url_object(self, url_object):
        goal_id = url_object.query_match.group('id')
        return f'{self.STARTREK_QUEUE}-{goal_id}'

    def _get_keys(self):
        return list(map(self.get_key_from_url_object, self.url_objects))

    def get_status_data(self, response):
        status = response['status']['key']
        return self.STATUS_DATA.get(status, ('Not known', '#888C91'))

    def get_importance_data(self, response):
        importance_key = settings.IMPORTANCE_KEYS.get(response['goalImportance']['id'], '')

        return self.IMPORTANCE_DATA.get(importance_key, self.IMPORTANCE_DATA['default'])

    def get_deadline(self, response):
        """
        Возвращает кортеж из срока
        для отображения и булевого значения,
        True - если дедлайн цели просрочен
        """
        deadline = response['deadline']
        overdue = False

        if not deadline:
            return '(Без срока)', overdue

        today = timezone.now()
        year = None
        deadline_datetime = datetime.datetime.strptime(deadline,
                                                       '%Y-%m-%d').replace(tzinfo=pytz.UTC)

        if today.year > deadline_datetime.year:
            year = '{} '.format(deadline_datetime.year)
        quarter = ((deadline_datetime.month - 1) // 3) + 1

        result = '({}Q{})'.format(year if year else '', quarter)

        if today > deadline_datetime:
            overdue = True
        return result, overdue

    def get_deadline_color(self, overdue):
        if overdue:
            return '#ff0000'
        return '#888C91'

    def add_result_for_object(self, response_data, url_object, data):
        key = self.get_key_from_url_object(url_object)
        url_data = response_data.get(key)
        if not url_data:
            data[url_object.url] = self.get_failed_result(url_object.url)
        else:
            title = url_data['summary']
            deadline, overdue = self.get_deadline(url_data)
            status, status_color = self.get_status_data(url_data)
            importance, importance_emoji = self.get_importance_data(url_data)
            responsible = url_data['assignee']['id']

            data[url_object.url] = List(
                ttl=self.get_ttl(key=status),
                value=[
                    Image(
                        src=self.FAVICON,
                        text=self.FAVICON_TEXT,
                    ),
                    String(value=title),
                    String(value=importance_emoji, hint=importance),
                    String(value=status, color=status_color),
                    String(value=deadline, color=self.get_deadline_color(overdue)),
                    User(login=responsible),
                ]
            )
