import asyncio
import functools
import logging
import urllib.parse

from concurrent.futures import ThreadPoolExecutor
from enum import Enum
from typing import Callable, TypeVar

from ids.registry import registry
from pydantic import BaseModel

from ..base import BaseWorker
from ...dto import Result, List, Inflect, UrlObject

logger = logging.getLogger(__name__)
T = TypeVar('T')


AVAILABLE_LANGUAGE = {'ru', 'en'}
DEFAULT_LANGUAGE = 'ru'
DEFAULT_PREFIX = 'кто'


class UserNotExists(Exception):
    pass


class Case(str, Enum):
    NOMINATIVE = 'им'
    GENITIVE = 'род'
    DATIVE = 'дат'
    ACCUSATIVE = 'вин'
    ABLATIVE = 'твор'
    PREPOSITIONAL = 'пр'


class Gender(str, Enum):
    MALE = 'm'
    FEMALE = 'f'


class InflectQuery(BaseModel):
    login: str
    lang: str = DEFAULT_LANGUAGE
    prefix: str = DEFAULT_PREFIX


CASE_PATTERN = {
    'кто': Case.NOMINATIVE,
    'укого': Case.GENITIVE,
    'ского': Case.GENITIVE,
    'кому': Case.DATIVE,
    'кого': Case.ACCUSATIVE,
    'кем': Case.ABLATIVE,
    'ком': Case.PREPOSITIONAL,
    'оком': Case.PREPOSITIONAL,
}


async def execute_async(executor: ThreadPoolExecutor, func: Callable[..., T], *args, **kwargs) -> T:
    loop = asyncio.get_event_loop()
    func_partial = functools.partial(func, *args, **kwargs)
    return await loop.run_in_executor(executor, func_partial)


class Worker(BaseWorker):
    HOSTNAME_REGEX = {'default': '^inflector$'}
    PATH_REGEX = r'^/(?P<login>[\w\-]+)/?$'  # начинается '/', содержит [\w, '-'] и может заканчиваться '/'

    TTL_MAP = {
        'default': 86400,  # 1 день
        'fail': 1200,  # 20 минут
    }

    async def get_result(self) -> Result:
        queries = [self.parse_url(url_object) for url_object in self.url_objects]
        tasks = [self.process_link(query=query) for query in queries]
        magic_links = await asyncio.gather(*tasks, return_exceptions=True)

        data = {}
        for url_object, magic_link, query in zip(self.url_objects, magic_links, queries):
            completed, ttl = True, self.TTL_MAP['default']

            if isinstance(magic_link, Exception):
                logger.exception(f'MAGICLINKS: Unhandled exception in worker {self.__class__.__name__}: {magic_link}')
                magic_link = Inflect(name=query.login)
                completed, ttl = False, self.TTL_MAP['fail']

            data[url_object.url] = List(value=[magic_link], ttl=ttl, completed=completed)

        return Result(data=data)

    async def process_link(self, query: InflectQuery) -> Inflect:
        staff = await self.request_staff(login=query.login)

        if query.lang not in AVAILABLE_LANGUAGE:
            query.lang, query.prefix = DEFAULT_LANGUAGE, DEFAULT_PREFIX

        case = CASE_PATTERN.get(query.prefix, Case.NOMINATIVE)
        name, last_name = await self.calc_name(staff, case=case, lang=query.lang)
        is_dismissed = False if staff.get('memorial', None) else staff['official']['is_dismissed']

        return Inflect(name=name, last_name=last_name, is_dismissed=is_dismissed)

    async def calc_name(self, staff: dict[str, dict], case: Case, lang: str) -> tuple[str, str]:
        name, last_name = staff['name']['first'][lang], staff['name']['last'][lang]

        if lang != 'ru' or case == Case.NOMINATIVE:
            return name, last_name

        gender = Gender.FEMALE if staff['personal']['gender'] == 'female' else Gender.MALE
        inflect = await self.request_inflect(name, last_name, gender=gender, case=case)

        return inflect['first_name'], inflect['last_name']

    async def request_staff(self, login: str) -> dict[str, ...]:
        staff_fields = 'name.first,name.last,personal.gender,official.is_dismissed,memorial.id'
        params = {'lookup': {'login': login, '_fields': staff_fields}}

        token = await self.get_token()
        repo = registry.get_repository('staff', 'person', user_agent=self.get_user_agent(), oauth_token=token)
        return await execute_async(executor=self.executor, func=repo.get_one, **params)

    async def request_inflect(self, first_name: str, last_name: str, gender: Gender, case: Case) -> dict[str, str]:
        params = {
            'case': case.value,
            'person': {
                'first_name': first_name,
                'last_name': last_name,
                'gender': gender.value,
            },
        }

        token = await self.get_token()
        repo = registry.get_repository('inflector', 'inflector', user_agent=self.get_user_agent(), oauth_token=token)
        return await execute_async(executor=self.executor, func=repo.inflect_person, **params)

    @staticmethod
    def parse_url(url_object: UrlObject) -> InflectQuery:
        query_param = urllib.parse.parse_qs(url_object.split_result.query)
        params = {key: value[0] for key, value in query_param.items()}
        params['login'] = url_object.path_match.group('login')
        return InflectQuery.parse_obj(params)
