import os

from django.conf import settings

from ..base import HttpSingleWorker
from ...dto.types import List, String, Image, BinaryResource


class Worker(HttpSingleWorker):
    HOSTNAME_REGEX = {'production': '^nanny.yandex-team.ru$',
                      'testing': '^dev-nanny.yandex-team.ru$',
                      'development': '^nanny.yandex-team.ru$',
                      }

    PATH_REGEX = '^/ui/$'
    FRAGMENT_REGEX = '^/t/(?P<key>[A-Z_]+-\d+)/?$'

    PATH = os.path.abspath(__file__)

    FAVICON = BinaryResource('favicon.ico', PATH)
    FAVICON_TEXT = 'Nanny'

    STRIKE_STATUSES = {
        'DEPLOY_SUCCESS',
        'CANCELLED',
    }

    STATUS_COLOR_MAP = {
        'IN_QUEUE':  '#80d8ff',
        'PREPARING':  '#00bcd4',
        'SWITCHING':  '#337ab7',
        'CANCELLED':  '#bdbdbd',
        'DEPLOY_SUCCESS':  '#4caf50',
        'DEPLOY_FAILED':  '#ff3d00',
        'ROLLED_BACK':  '#ff6d00',
        'ON_HOLD':  '#ffd740',
        'COMMITTED':  '#00bfa5',
        'PREPARED':  '#8bc34a',
    }
    DEFAULT_COLOR = '#888C91'

    TTL_MAP = {
        'default': 60,  # 1 минуту
        'fail': 60,
        'DEPLOY_SUCCESS': 3600,  # 1 час
        'CANCELLED': 3600,
    }

    async def get_token(self):
        return settings.NANNY_TOKEN

    def get_request_url(self, url_object, api_name=None):
        ticket_key = url_object.fragment_match.group('key')
        return 'https://{}/api/tickets/GetTicket/?id={}'.format(self.api_url,
                                                                ticket_key,
                                                                )

    def get_color_for_status(self, status):
        return self.STATUS_COLOR_MAP.get(status, self.DEFAULT_COLOR)

    def get_service(self, data):
        service_dev = data['spec'].get('serviceDeployment')
        service = '-'
        if service_dev:
            service = service_dev['serviceId']
        return service

    def get_strike(self, status):
        return status in self.STRIKE_STATUSES

    def handle_successful_response(self, url_object, response):
        response = response['value']
        ticket_key = response['id']
        status = response['status']['status']
        service = self.get_service(response)

        result = List(
            ttl=self.get_ttl(key=status),
            value=[
                Image(
                    src=self.FAVICON,
                    text=self.FAVICON_TEXT,
                ),
                String(value=ticket_key,
                       strike=self.get_strike(status),
                       ),
                String(value=service),
                String(value=status,
                       color=self.get_color_for_status(status),
                       ),
            ]
        )

        return result
