import json
import os

from ..base import HttpSingleWorker
from ...dto.types import List, String, Image, User, BinaryResource
from ...utils.response import is_failed_response


class Worker(HttpSingleWorker):
    method = 'POST'

    HOSTNAME_REGEX = {'default': r'^(beta\.)?nirvana.yandex-team.ru$',
                      }
    PATH_REGEX = r'''(?x)/(?P<type>flow|process)/(?P<id>[a-zA-Z0-9-]+)/?
                     ((?P<instance_id>(?!(graph|options))[a-zA-Z0-9-]+?))?
                     (/(graph|options))?(/)?(?P<block>(FlowchartBlockOperation|operation)/)?
                     (?(block)(?P<block_id>[a-zA-Z0-9-]+))/?$'''

    STATUS_COLOR_MAP = {
        'success': '#3dbf41',
        'failure': '#ff0000',
        'running': '#3dbf41',
        'canceled': '#eabf3d',
        'not started': '#999999'
    }
    DEFAULT_COLOR = '#888C91'

    PATH = os.path.abspath(__file__)

    FAVICON = BinaryResource('favicon.ico', PATH)
    FAVICON_TEXT = 'Nirvana'

    TTL_MAP = {
        'default': 1800,  # 30 минут
        'fail': 60,
        'success': 604800,  # 1 неделя
        'failure': 604800,
        'canceled': 604800,
    }

    async def get_headers(self):
        headers = await super().get_headers()
        headers.update({"Content-Type": "application/json"})
        return headers

    def get_params(self, url_object):
        instance_id = url_object.path_match.group('instance_id')

        if instance_id:
            params = {
                "workflowInstanceId": instance_id
            }
        elif url_object.path_match.group('type') == 'flow':
            params = {
                "workflowId": url_object.path_match.group('id')
            }
        elif url_object.path_match.group('type') == 'process':
            params = {
                "workflowInstanceId": url_object.path_match.group('id')
            }

        if url_object.path_match.group('block_id'):
            params["blocks"] = [{"guid": url_object.path_match.group('block_id')}]
        return params

    def get_request_data(self, url_object, api_name=None, params=None):
        request_data = dict(jsonrpc='2.0',
                            method='getWorkflowSummary',
                            id='someId',
                            params=self.get_params(url_object),
                            )
        return json.dumps(request_data)

    def get_request_url(self, url_object, api_name=None):
        return 'https://nirvana.yandex-team.ru/api/public/v1/getWorkflowSummary'

    def get_graph_name(self, data, url_object):
        """
        :type data: dict
        :return: basestring
        """
        graph_name = data['result'].get('workflowName')
        if not graph_name:
            graph_name = url_object.path_match.group('instance_id') or url_object.path_match.group('id')
        block_data = data['result']['blockSummaries']
        if block_data:
            block_name = block_data[0]['blockName']
            graph_name += ' \u2192 {}'.format(block_name)

        return graph_name

    def get_graph_status(self, data):
        """
        :type data: dict
        :return: basestring
        """
        result = data['result']['result']
        status = data['result']['status']
        if result == 'undefined':
            if status == 'undefined':
                return 'not started'
            return status
        elif result == 'cancel':
            return 'canceled'
        return result

    def handle_response(self, url_object, response):
        if is_failed_response(response) or response.get('error'):
            return self.handle_failed_response(url_object, response)
        return super().handle_response(url_object, response)

    def handle_successful_response(self, url_object, response):
        graph_name = self.get_graph_name(response, url_object)
        graph_status = self.get_graph_status(response)
        graph_owner = response['result']['workflowOwner']

        result = List(
            ttl=self.get_ttl(key=graph_status),
            value=[
                Image(
                    src=self.FAVICON,
                    text=self.FAVICON_TEXT,
                ),
                String(value=graph_status.title(),
                       color=self.STATUS_COLOR_MAP.get(graph_status, self.DEFAULT_COLOR)),
                String(value=graph_name),
                User(login=graph_owner),
            ]
        )

        return result
