import os

from ids.registry import registry

from ..base import IdsBulkWorker
from ...dto import Result, List, Image, String, BinaryResource


class Worker(IdsBulkWorker):
    HOSTNAME_REGEX = {'production': '^staff.yandex-team.ru$',
                      'testing': '^staff.test.yandex-team.ru$',
                      'development': '^staff.yandex-team.ru$',
                      }
    PATH_REGEX = '^/departments/(?P<url>\w+)/?$'

    PATH = os.path.abspath(__file__)

    FAVICON = BinaryResource('favicon.ico', PATH)
    FAVICON_TEXT = 'Staff'

    TTL_MAP = {
        'default': 86400,  # 1 день
        'fail': 60,
    }

    async def get_repository(self):
        token = await self.get_token()
        return registry.get_repository('staff', 'group', user_agent=self.get_user_agent(),
                                       oauth_token=token, **self.get_request_params())

    def get_lookup_data(self):
        return {
            'type': 'department',
            'url': ','.join(url_object.path_match.group('url') for url_object in self.url_objects),
            '_fields': 'url,name,affiliation_counters',
        }

    def get_prepared_data(self, response):
        return {department_data['url']: department_data
                for department_data in response}

    def get_failed_value(self, url_object):
        return url_object.path_match.group('url')

    def handle_successful_response(self, response):
        data = {}
        prepared_data = self.get_prepared_data(response)
        for url_object in self.url_objects:
            url_object_url = url_object.path_match.group('url')
            url_data = prepared_data.get(url_object_url)
            if url_data:
                response_data = (
                    String(value=url_data['name']),
                    String(value=sum(url_data['affiliation_counters'].values()), color='gray'),
                )
            else:
                response_data = (String(value=url_object.url),
                                 )
            url_response = List(
                ttl=self.get_ttl(),
                value=[
                    Image(
                        src=self.FAVICON,
                        text=self.FAVICON_TEXT,
                    ),
                ])
            for value in response_data:
                url_response.value.append(value)

            data[url_object.url] = url_response
        return Result(data=data)
