# coding: utf-8
import os
import logging

from ids.registry import registry

from ..base import IdsBulkWorker
from ...dto import Result, List, Image, String, BinaryResource


logger = logging.getLogger(__name__)


class Worker(IdsBulkWorker):
    HOSTNAME_REGEX = {'production': '^staff.yandex-team.ru$',
                      'testing': '^staff.test.yandex-team.ru$',
                      'development': '^staff.yandex-team.ru$',
                      }
    PATH_REGEX = '^/map/table/(?P<number>\d+)/?$'

    PATH = os.path.abspath(__file__)

    FAVICON = BinaryResource('favicon.ico', PATH)
    FAVICON_TEXT = 'Staff'

    TTL_MAP = {
        'default': 86400,  # 1 день
        'fail': 60,
    }

    async def get_repository(self):
        token = await self.get_token()
        return registry.get_repository('staff', 'table', user_agent=self.get_user_agent(),
                                       oauth_token=token, **self.get_request_params())

    def get_lookup_data(self):
        return {
            'id': ','.join(url_object.path_match.group('number') for url_object in self.url_objects),
            '_fields': 'id,floor.name.ru,floor.office.name.ru',
        }

    def get_failed_value(self, url_object):
        return url_object.path_match.group('number')

    def get_prepared_data(self, response):
        return {str(table_data['id']): table_data
                for table_data in response}

    def handle_successful_response(self, response):
        data = {}
        prepared_data = self.get_prepared_data(response)
        for url_object in self.url_objects:
            url_object_number = url_object.path_match.group('number')
            url_data = prepared_data.get(str(url_object_number))
            if url_data:
                response_data = (
                    String(value="{},".format(url_data['floor']['office']['name']['ru'])),
                    String(value="{},".format(url_data['floor']['name']['ru'])),
                    Image(src=self.FAVICON, text=self.FAVICON_TEXT),
                    String(value=url_object_number),
                )
            else:
                response_data = (String(value=url_object.url),)

            url_response = List(
                ttl=self.get_ttl(),
                value=list(response_data))

            data[url_object.url] = url_response

        return Result(data=data)
