import os
import logging

from .startrek_base import StartrekBase
from ...dto import List, Image, String, User, BinaryResource

logger = logging.getLogger(__name__)


class Worker(StartrekBase):
    HOSTNAME_REGEX = {'production': '^st.yandex-team.ru$',
                      'testing': '^st.test.yandex-team.ru$',
                      'development': '^st.yandex-team.ru$',
                      }
    PATH_REGEX = r'^/(?P<key>[A-Z]+-\d+)/?$'

    TICKET_FIELDS = 'assignee', 'summary', 'status', 'aliases', 'resolution',

    PATH = os.path.abspath(__file__)

    FAVICON = BinaryResource('favicon.ico', PATH)
    FAVICON_TEXT = 'Startrek'

    strike_color = '#999'

    TTL_MAP = {
        'default': 60,  # 1 минута
        'fail': 60,
    }

    def check_objects_count(self):
        objects_count = len(self.url_objects)
        if objects_count > 1000:
            referer = self.request.META.get('HTTP_REFERER', 'No data')
            logger.warning('Got {} links from {}'.format(objects_count, referer))
        return objects_count

    def get_timeout(self):
        # стартреку плохо, и у него подросли тайминги
        # лучше подождать иначе мало того что мы повысим RPS
        # но и пользователь увидет ответ лишь на следующей итерации
        return 1.5

    def _get_keys(self):
        def get_key_from_url_object(url_object):
            return url_object.path_match.group('key')

        return list(map(get_key_from_url_object, self.url_objects))

    def get_strike(self, url_data):
        return url_data.get('resolution') is not None

    def add_result_for_object(self, response_data, url_object, data):
        key = url_object.path_match.group('key')
        url_data = response_data.get(key)
        if not url_data:
            data[url_object.url] = self.get_failed_result(key)
        else:
            ticket_key = url_data['key']
            status_data = url_data['status']
            status = status_data['display']
            strike = self.get_strike(url_data)
            color_params = {'color': self.strike_color} if strike else {}
            data[url_object.url] = List(
                ttl=self.get_ttl(),
                value=[
                    Image(
                        src=self.FAVICON,
                        text=self.FAVICON_TEXT,
                    ),
                    String(value=ticket_key,
                           strike=strike,
                           action={
                               "event": "click",
                               "type": "halfscreenpreview",
                               "url": self.get_issue_url(url_object.hostname_match.group(),
                                                         ticket_key,
                                                         )
                           },
                           **color_params),
                    String(value=status, color='#999'),
                    String(value=url_data['summary']),
                ]
            )
            self.add_assignee(data=data, url_object=url_object, url_data=url_data)

    def add_assignee(self, data, url_object, url_data):
        assignee_data = url_data.get('assignee')
        if assignee_data is not None:
            data[url_object.url].value.append(User(login=assignee_data['id']))

    def get_issue_url(self, st_host, st_issue):
        return "https://{}/issueCard/{}".format(st_host, st_issue)

    def get_failed_value(self, url_object):
        return url_object.path_match.group('key')
