import datetime
import os

from django.conf import settings

from .abc_duty import Worker as ABCWorker
from ..mixins.person_ticket import PersonTicketMixin
from ...dto.types import List, String, Image, User, BinaryResource


class Worker(PersonTicketMixin, ABCWorker):
    TVM2_DESTINATION_CLIENT_ID = settings.WATCHER_TVM2_CLIENT_ID

    PATH_REGEX = r'/services/(?P<service>[-\w]+)/duty2/(?P<schedule_id>[\d]+)$'
    QUERY_REGEX = ''

    PATH = os.path.abspath(__file__)

    FAVICON = BinaryResource('favicon.ico', PATH)
    FAVICON_TEXT = 'Watcher'

    def get_timeout(self):
        return 0.8

    def get_request_url(self, url_object, api_name=None):
        schedule_id = url_object.path_match.group('schedule_id')
        return 'https://{}/api/watcher/v1/shift/all_fields?filter=schedule_id={},status=active'.format(
            settings.ABC_API_URL, schedule_id
        )

    def format_date(self, response):
        date_parsed = datetime.datetime.fromisoformat(response['end'])
        hour_minute = ''
        if date_parsed.hour or date_parsed.minute:
            hour_minute = ' {:02}:{:02}'.format(date_parsed.hour, date_parsed.minute)
        return '(до{} {}-го {})'.format(
            hour_minute, date_parsed.day, self.MONTH_MAP[date_parsed.month]
        )

    def collect_duty(self, shifts):
        primary_users = []
        backup_users = []
        is_empty = True

        for shift in shifts:
            if not shift:
                continue
            is_empty = is_empty & shift.get('empty', False)

            staff = shift.get('staff', None)
            if not staff:
                continue

            login = staff.get('login', None)
            if not login:
                continue

            is_primary = shift.get('is_primary', True)
            if is_primary:
                primary_users.append(User(login=login))
            else:
                backup_users.append(User(login=login))

        duty_list = None
        if primary_users:
            if not backup_users:
                duty_list = primary_users
            else:
                duty_list = []
                for users, suffix in (
                    (primary_users, String(value='(primary)')),
                    (backup_users, String(value='(backup)'))
                ):
                    for user in users:
                        duty_list.append(List(
                            ttl=self.get_ttl(),
                            value=[user, suffix],
                            separator=' ',
                        ))
        elif backup_users:
            duty_list = backup_users

        result = None
        if duty_list:
            result = List(ttl=self.get_ttl(), value=duty_list, separator=', ')

        return result, is_empty

    def handle_successful_response(self, url_object, response):
        if 'result' not in response or not len(response['result']):
            return self.handle_failed_response(url_object, response, completed=True)

        shifts = response['result']
        schedule_name = shifts[0]['schedule']['name']
        end_datetime = self.format_date(shifts[0])
        collected_duty, is_empty = self.collect_duty(shifts)

        if collected_duty:
            duty_string = collected_duty
        elif is_empty:
            duty_string = String(value='Никто не дежурит', color='gray')
        else:
            duty_string = String(value='Никто не дежурит', color='red')

        result = List(
            ttl=self.get_ttl(),
            value=[
                Image(src=self.FAVICON, text=self.FAVICON_TEXT),
                String(value=schedule_name),
                duty_string,
                String(value=end_datetime, color='#999999')
            ]
        )
        return result
