import os
import re
import yenv
from django.conf import settings

from django.utils.functional import cached_property

from ..base import HttpBulkWorker
from ...dto import Result, List, Image, String, User, BinaryResource
from ..mixins.person_ticket import PersonTicketMixin


class Worker(PersonTicketMixin, HttpBulkWorker):
    HOSTNAME_REGEX = {'production': '^procu.yandex-team.ru$',
                      'testing': '^procu.test.yandex-team.ru$',
                      'development': '^procu.test.yandex-team.ru$',
                      }
    PATH_REGEX = r'^/(?P<key>YP\d+)(?:/(?:procurement|comments)?)?/?$'

    PATH = os.path.abspath(__file__)

    FAVICON = BinaryResource('favicon.ico', PATH)
    FAVICON_TEXT = 'Zakupki'

    STATUS_COLOR_MAP = {
        'closed': '#ee0000',
        'shipped': '#880088',
        'checkout': '#008800',
        'review': '#00aaff',
        'bidding': '#eeaa00',
        'draft': '#666666',
    }
    DEFAULT_COLOR = '#888C91'

    method = 'POST'

    TTL_MAP = {
        'default': 300,  # 5 минут
        'fail': 60,  # 1 минута
        'closed': 3600,  # 1 час
    }

    TVM2_DESTINATION_CLIENT_ID = settings.ZAKUPKI_TVM2_CLIENT_ID

    @cached_property
    def api_host(self):
        host_regex = re.match(r'\^(?P<host>.+)\$', self.HOSTNAME_REGEX[yenv.type])
        return host_regex.group('host')

    def get_request_url(self):
        return 'https://{}/api/wiki/magiclink'.format(self.api_host)

    def get_request_json(self):
        return {
            "urls": [
                url_object.url for url_object in
                self.url_objects
            ],
        }

    def get_color_for_status(self, status):
        return self.STATUS_COLOR_MAP.get(status, self.DEFAULT_COLOR)

    def get_prepared_data(self, response):
        response_data = {
            item['key']: item
            for item in response
            if item.get('status')
        }
        return response_data

    def get_status_data(self, data):
        status_data = data['status']
        return status_data['key'], status_data['name']

    def get_strike_for_key(self, status):
        return status == 'closed'

    def get_failed_value(self, url_object):
        return url_object.path_match.group('key')

    def handle_successful_response(self, response):
        data = {}
        response_data = self.get_prepared_data(response)
        for url_object in self.url_objects:
            url_key = url_object.path_match.group('key')
            url_data = response_data.get(url_key)
            if not url_data:
                data[url_object.url] = self.get_failed_result(url_key)
            else:
                status_key, status_name = self.get_status_data(url_data)

                data[url_object.url] = List(
                    ttl=self.get_ttl(key=status_key),
                    value=[
                        Image(
                            src=self.FAVICON,
                            text=self.FAVICON_TEXT,
                        ),
                        String(value=url_key,
                               strike=self.get_strike_for_key(status_key),
                               ),
                        String(value=status_name,
                               color=self.get_color_for_status(status_key),
                               ),
                        String(value=url_data['title']),
                    ]
                )
                user_login = url_data.get('username')
                if user_login is not None:
                    data[url_object.url].value.append(User(login=user_login))
        return Result(data=data)
