import os

from django.conf import settings

from abc import abstractmethod

from django.utils.functional import cached_property

from ...dto import Image, BinaryResource, Result


class FemidaMixin:

    HOSTNAME_REGEX = {
        'production': '^femida.yandex-team.ru$',
        'testing': '^femida.test.yandex-team.ru$',
        'development': '^femida.yandex-team.ru$',
    }

    PATH = os.path.abspath(__file__)

    FAVICON = BinaryResource('favicon.ico', PATH)
    FAVICON_TEXT = 'Femida'

    TTL_MAP = {
        'default': 1800,  # 30 минут
        'fail': 60,
        'problems': 86400,  # 1 день
        'candidates': 604800,  # 1 неделя
    }

    TVM2_DESTINATION_CLIENT_ID = settings.FEMIDA_TVM2_CLIENT_ID

    @property
    def favicon(self):
        return Image(
            src=self.FAVICON,
            text=self.FAVICON_TEXT,
        )

    @cached_property
    def interview_grade_icons(self):
        return {
            grade: BinaryResource('grade{}.ico'.format(grade), self.PATH)
            for grade in range(9)
        }


class FemidaBulkMixin(FemidaMixin):

    RESOURCE = None
    FIELDS = ()

    def get_request_url(self):
        # TODO: ходить в несколько запросов по N объектов
        ids = [u.path_match.group('id') for u in self.url_objects]
        fields = ('id',) + self.FIELDS
        return 'https://{host}/_api/magiclinks/{resource}/?id={ids}&_fields={fields}'.format(
            host=self.api_url,
            resource=self.RESOURCE,
            ids=','.join(ids),
            fields=','.join(fields),
        )

    def get_prepared_data(self, response):
        return {str(i['id']): i for i in response['results']}

    @abstractmethod
    def handle_item(self, item):
        pass

    def handle_successful_response(self, response):
        prepared_data = self.get_prepared_data(response)
        data = {}
        for url_object in self.url_objects:
            item_id = url_object.path_match.group('id')
            if item_id in prepared_data:
                data[url_object.url] = self.handle_item(prepared_data[item_id])
            else:
                value = self.get_failed_value(url_object)
                data[url_object.url] = self.get_failed_result(value=value)
        return Result(data=data)
