import os

import html2text

from ...dto import Result, List, Image, String, BinaryResource, User


class SandboxMixin:
    HOSTNAME_REGEX = {'production': '^sandbox.yandex-team.ru$',
                      'testing': '^www-sandbox1.n.yandex-team.ru$',
                      'development': '^sandbox.yandex-team.ru$',
                      }

    BASE_API_URL = ('https://{host}/api/v1.0/{resource_type}?'
                    'id={ids}&limit={limit}')

    STATUS_FIELD_NAME = 'status'

    METHOD = 'GET'

    PATH = os.path.abspath(__file__)
    FAVICON = BinaryResource('favicon.ico', PATH)
    FAVICON_TEXT = 'Sandbox'

    DEFAULT_COLOR = '#888C91'

    TTL_MAP = {
        'default': 1800,  # 30 минут
        'fail': 60,  # 1 минута
    }

    def get_urls_data(self):
        ids = [url_object.path_match.group('key') for
               url_object in self.url_objects]
        return ','.join(ids), len(ids)

    def get_request_url(self):
        ids, limit = self.get_urls_data()
        return self.BASE_API_URL.format(host=self.api_url,
                                        resource_type=self.SANDBOX_TYPE,
                                        ids=ids,
                                        limit=limit,
                                        )

    def get_prepared_data(self, response):
        return {
            str(item['id']): item
            for item in response['items']
        }

    def get_color_for_status(self, status):
        return self.STATUS_COLOR_MAP.get(status, self.DEFAULT_COLOR)

    def get_owner_response(self, url_data):
        owner_login = url_data['owner']
        if owner_login.islower():
            response = User(login=owner_login)
        else:
            response = List(ttl=self.get_ttl(),
                            separator='',
                            value=[
                                String(value=owner_login[0], color='red'),
                                String(value=owner_login[1:], color='#888C91'),
                            ])
        return response

    def get_description(self, url_data):
        return url_data.get('description', '')

    def get_description_response(self, url_data):
        description = self.get_description(url_data)
        parsed_description = html2text.html2text(description).strip('\n')
        if len(parsed_description) > 255:
            parsed_description = '{}...'.format(parsed_description[:255])
        return parsed_description

    def get_type(self):
        return self.SANDBOX_TYPE[0].upper()

    def handle_successful_response(self, response):
        data = {}
        prepared_data = self.get_prepared_data(response)
        for url_object in self.url_objects:
            url_key = url_object.path_match.group('key')
            url_data = prepared_data.get(url_key)
            if url_data:
                status = url_data.get(self.STATUS_FIELD_NAME, '')
                owner_response = self.get_owner_response(url_data)
                response_data = [
                    String(value='#{}'.format(url_key),
                           color='#ABABAB',
                           action={
                               "event": "click",
                               "type": "halfscreenpreview",
                               "url": url_object.url
                           }),
                    String(value='({})'.format(self.get_type())),
                    String(value=status,
                           color=self.get_color_for_status(status)),
                    String(value=self.get_description_response(url_data)),
                    owner_response,
                ]
            else:
                response_data = (
                    String(value=url_object.url),
                )
            url_result = List(
                ttl=self.get_ttl(),
                value=[
                    List(ttl=self.get_ttl(),
                         value=[
                             Image(
                                 src=self.FAVICON,
                                 text=self.FAVICON_TEXT,
                             )],
                         action={
                             "event": "click",
                             "type": "halfscreenpreview",
                             "url": url_object.url,
                         }),
                ])
            for value in response_data:
                url_result.value.append(value)

            data[url_object.url] = url_result

        return Result(data=data)
