import logging
import traceback
from functools import wraps

from django.conf import settings
from django.core.exceptions import PermissionDenied
from django.http import JsonResponse
from django.utils.deprecation import MiddlewareMixin

from django_yauth.middleware import YandexAuthBackendMiddleware
from django_yauth import util

from intranet.magiclinks.src.links.utils.auth import is_tvm_authentication

logger = logging.getLogger(__name__)


class ExceptionMiddleware(MiddlewareMixin):
    def process_exception(self, request, exception):
        logger.exception('MAGICLINKS: Unhandled exception: %s', exception)
        if settings.DEBUG:
            return JsonResponse({
                'error': {
                    'message': str(exception),
                    'traceback': traceback.format_exc()
                }
            })


class YauthMiddleware(YandexAuthBackendMiddleware):

    def process_view(self, request, view_func, view_args, view_kwargs):
        if getattr(view_func, 'yauth_check_exempt', False):
            return

        if not is_valid_auth(request):
            raise PermissionDenied('Invalid auth data')

        request.user_ip = util.get_real_ip(request)


def yauth_check_exempt(func):
    @wraps(func)
    def wrapped(*args, **kwargs):
        return func(*args, **kwargs)

    wrapped.yauth_check_exempt = True
    return wrapped


def is_valid_auth(request):
    if not request.yauser:
        return False

    if is_tvm_authentication(request):
        # В случае TVM аутентификации пользовательского тикета может не быть
        # если пользователь был авторизован через облачный сервис AUTH,
        # но в этом случае в заголовках должены быть переданы
        # x-uid или x-cloud-uid с id паспортного или облачного пользователя
        # и x-org-id с id организации (у нас нет пользовальской куки и мы не можем
        # сами определить в какой организации в данный момент залогинен пользователь)
        if request.yauser.raw_user_ticket:
            return True

        org_id = request.META.get('HTTP_X_ORG_ID')
        if not org_id:
            logger.error('auth error: x-org-id is required')
            return False

        request.org_id = org_id

        uid = request.META.get('HTTP_X_UID')
        if uid:
            request.yauser.uid = uid
            return True

        cloud_uid = request.META.get('HTTP_X_CLOUD_UID')
        if cloud_uid:
            request.yauser.uid = cloud_uid
            request.is_cloud_user = True
            return True

        logger.error('auth error: x-uid or x-cloud-uid is required')
        return False

    return True
