import copy
from functools import partial

import sentry_sdk
from django_pgaas import HostManager
from django_replicated.settings import *  # noqa F403
from errorboosterclient.sentry import ErrorBoosterTransport
from sentry_sdk.integrations.celery import CeleryIntegration

from mentor.contrib.logbroker.logbroker import send_me
from mentor.contrib.sentry.integrations.django import (
    CustomDjangoIntegration as DjangoIntegration,
)

from .base import *  # noqa F403
from .base import env

# GENERAL
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#secret-key
SECRET_KEY = env("DJANGO_SECRET_KEY")
# https://docs.djangoproject.com/en/dev/ref/settings/#allowed-hosts
ALLOWED_HOSTS = env.list("DJANGO_ALLOWED_HOSTS", default=["*"])

# DATABASES
# ------------------------------------------------------------------------------
POSTGRES_HOST_JSON = env.json("POSTGRES_HOST_JSON", default={})
DEPLOY_NODE_DC = env("DEPLOY_NODE_DC", default="myt")
if POSTGRES_HOST_JSON:
    host_manager = HostManager(
        list(POSTGRES_HOST_JSON.items()),
        current_dc=DEPLOY_NODE_DC,
    )
    DATABASES["default"]["HOST"] = host_manager.host_string
else:
    DATABASES["default"]["HOST"] = env("POSTGRES_HOST")

DATABASES["default"]["PORT"] = env.int("POSTGRES_PORT", default="6432")
DATABASES["default"]["ATOMIC_REQUESTS"] = True
DATABASES["default"]["OPTIONS"] = {
    "sslmode": "verify-full",
    "sslrootcert": "/app/.postgres/root.crt",
    "target_session_attrs": "read-write",
}
DATABASES["default"]["DISABLE_SERVER_SIDE_CURSORS"] = True

DATABASES["slave"] = copy.deepcopy(DATABASES["default"])
DATABASES["slave"]["OPTIONS"]["target_session_attrs"] = "any"

# django-replicated
# ------------------------------------------------------------------------------
REPLICATED_DATABASE_SLAVES = ["slave"]
DATABASE_ROUTERS = ["django_replicated.router.ReplicationRouter"]
REPLICATED_VIEWS_OVERRIDES = {
    "/admin/*": "master",
}

# django_pgaas
# ------------------------------------------------------------------------------
PGAAS_USE_ZDM = True

# redis-sentinel
# ------------------------------------------------------------------------------
REDIS_SENTINELS = env.json("REDIS_SENTINELS", default=[])
REDIS_SERVICE_NAME = env("REDIS_SERVICE_NAME", default="mentor")
REDIS_PASSWORD = env("REDIS_PASSWORD", default=None)

REDIS_MAX_CONNECTIONS = env("REDIS_MAX_CONNECTIONS", default=100)
DJANGO_REDIS_CONNECTION_FACTORY = "django_redis.pool.SentinelConnectionFactory"

# CACHES
# ------------------------------------------------------------------------------
CACHES = {
    "default": {
        "BACKEND": "django_redis.cache.RedisCache",
        "LOCATION": "redis://{service_name}/{db}".format(
            service_name=REDIS_SERVICE_NAME, db=0
        ),
        "OPTIONS": {
            "CLIENT_CLASS": "django_redis.client.SentinelClient",
            "CONNECTION_POOL_KWARGS": {
                "max_connections": REDIS_MAX_CONNECTIONS,
                "retry_on_timeout": True,
            },
            "SENTINELS": REDIS_SENTINELS,
            "PASSWORD": REDIS_PASSWORD,
        },
    }
}

# SECURITY
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#secure-proxy-ssl-header
SECURE_PROXY_SSL_HEADER = ("HTTP_X_FORWARDED_PROTO", "https")
# https://docs.djangoproject.com/en/dev/ref/settings/#secure-ssl-redirect
SECURE_SSL_REDIRECT = env.bool("DJANGO_SECURE_SSL_REDIRECT", default=True)
# https://docs.djangoproject.com/en/dev/ref/settings/#session-cookie-secure
SESSION_COOKIE_SECURE = True
# https://docs.djangoproject.com/en/dev/ref/settings/#csrf-cookie-secure
CSRF_COOKIE_SECURE = True
# https://docs.djangoproject.com/en/dev/topics/security/#ssl-https
# https://docs.djangoproject.com/en/dev/ref/settings/#secure-hsts-seconds
# TODO: set this to 60 seconds first and then to 518400 once you prove the former works
SECURE_HSTS_SECONDS = 60
# https://docs.djangoproject.com/en/dev/ref/settings/#secure-hsts-include-subdomains
SECURE_HSTS_INCLUDE_SUBDOMAINS = env.bool(
    "DJANGO_SECURE_HSTS_INCLUDE_SUBDOMAINS", default=True
)
# https://docs.djangoproject.com/en/dev/ref/settings/#secure-hsts-preload
SECURE_HSTS_PRELOAD = env.bool("DJANGO_SECURE_HSTS_PRELOAD", default=True)
# https://docs.djangoproject.com/en/dev/ref/middleware/#x-content-type-options-nosniff
SECURE_CONTENT_TYPE_NOSNIFF = env.bool(
    "DJANGO_SECURE_CONTENT_TYPE_NOSNIFF", default=True
)

# STATIC
# ------------------------
STATICFILES_STORAGE = "whitenoise.storage.CompressedManifestStaticFilesStorage"

# MEDIA
# ------------------------------------------------------------------------------

# TEMPLATES
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#templates
TEMPLATES[-1]["OPTIONS"]["loaders"] = [  # type: ignore[index] # noqa F405
    (
        "django.template.loaders.cached.Loader",
        [
            "django.template.loaders.filesystem.Loader",
            "django.template.loaders.app_directories.Loader",
        ],
    )
]

# EMAIL
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#default-from-email
DEFAULT_FROM_EMAIL = env(
    "DJANGO_DEFAULT_FROM_EMAIL", default="mentor <noreply@mentor.yandex-team.ru>"
)
# https://docs.djangoproject.com/en/dev/ref/settings/#server-email
SERVER_EMAIL = env("DJANGO_SERVER_EMAIL", default=DEFAULT_FROM_EMAIL)
# https://docs.djangoproject.com/en/dev/ref/settings/#email-subject-prefix
EMAIL_SUBJECT_PREFIX = env("DJANGO_EMAIL_SUBJECT_PREFIX", default="[mentor]")


# LOGGING
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#logging
# See https://docs.djangoproject.com/en/dev/topics/logging for
# more details on how to customize your logging configuration.
# A sample logging configuration. The only tangible logging
# performed by this configuration is to send an email to
# the site admins on every HTTP 500 error when DEBUG=False.
LOGGING = {
    "version": 1,
    "disable_existing_loggers": True,
    "formatters": {
        "json": {
            "()": "mentor.contrib.logger.format.CustomQloudJsonFormatter",
        },
    },
    "handlers": {
        "console": {
            "level": "DEBUG",
            "class": "logging.StreamHandler",
            "formatter": "json",
        }
    },
    "root": {"level": "INFO", "handlers": ["console"]},
    "loggers": {
        "": {
            "handlers": ["console"],
            "level": "WARNING",
            "propagate": False,
        },
        "django": {
            "handlers": ["console"],
            "level": "INFO",
            "propagate": False,
        },
        "django.db.backends": {
            "level": "ERROR",
            "handlers": ["console"],
            "propagate": False,
        },
        "celery": {
            "handlers": ["console"],
            "level": "INFO",
            "propagate": True,
        },
        "gunicorn.error": {
            "level": "INFO",
            "handlers": ["console"],
            "propagate": False,
        },
        "gunicorn.access": {
            "level": "INFO",
            "handlers": ["console"],
            "propagate": False,
        },
        "django_replicated": {
            "handlers": ["console"],
            "level": "INFO",
            "propagate": False,
        },
        "django_celery_beat": {
            "level": "INFO",
            "handlers": ["console"],
            "propagate": False,
        },
        # Errors logged by the SDK itself
        "sentry_sdk": {
            "level": "ERROR",
            "handlers": ["console"],
            "propagate": False,
        },
        "django.security.DisallowedHost": {
            "level": "ERROR",
            "handlers": ["console"],
            "propagate": False,
        },
    },
}

# Celery
# ------------------------------------------------------------------------------
CELERY_BROKER_URL = env("CELERY_BROKER_URL")

CELERY_BROKER_TRANSPORT_OPTIONS = {
    "sentinels": REDIS_SENTINELS,
    "service_name": REDIS_SERVICE_NAME,
    "db": 0,
    "password": REDIS_PASSWORD,
}

# ERROR BOOSTER
# ------------------------------------------------------------------------------
ERROR_BOOSTER_ENABLED = env("ERROR_BOOSTER_ENABLED", default=True)

if ERROR_BOOSTER_ENABLED:
    LOGBROKER_TOKEN = env("LOGBROKER_TOKEN")
    LOGBROKER_SOURCE = env("LOGBROKER_SOURCE")
    LOGBROKER_TOPIC = env("LOGBROKER_TOPIC")
    ERROR_BOOSTER_PROJECT = env("ERROR_BOOSTER_PROJECT")
    ERROR_BOOSTER_ENVIRONMENT = env("ERROR_BOOSTER_ENVIRONMENT")
    LOGBROKER_DATACENTER = env("LOGBROKER_DATACENTER", default="myt")
    LOGBROKER_PORT = env.int("LOGBROKER_PORT", default=2135)

    sentry_sdk.init(
        transport=ErrorBoosterTransport(
            project=ERROR_BOOSTER_PROJECT,  # Псевдоним проекта в бустере.
            sender=partial(
                send_me,
                token=LOGBROKER_TOKEN,
                source=LOGBROKER_SOURCE,
                topic=LOGBROKER_TOPIC,
            ),
        ),
        environment=ERROR_BOOSTER_ENVIRONMENT,
        # Если нужно подкрутить время, выделенное
        # на работу фоновой отправляющей нити.
        shutdown_timeout=20,
        # Для интеграции с Django:
        integrations=[DjangoIntegration(), CeleryIntegration()],
        send_default_pii=True,
    )

# TVM
# ------------------------------------------------------------------------------
MIDDLEWARE += [
    "mentor.contrib.blackbox.middleware.AuthMiddleware",
    "mentor.users.middleware.UserPresenceMiddleware",
]

# Staff
# ------------------------------------------------------------------------------
STAFF_API_AUTH_METHOD = env("STAFF_API_AUTH_METHOD", default="OAuth")

# ElasticSearch
# ------------------------------------------------------------------------------
ELASTICSEARCH_USE_SSL = env.bool("ELASTICSEARCH_USE_SSL", default=True)
ELASTICSEARCH_CA_CERTS = env(
    "ELASTICSEARCH_CA_CERTS", default="/etc/ssl/certs/ca-certificates.crt"
)

ELASTICSEARCH_API_KEY = env("ELASTICSEARCH_API_KEY", default="")

ELASTICSEARCH_DSL = {
    "default": {
        "hosts": env.list("ELASTICSEARCH_HOSTS", default=[]),
        "port": 9200,
        "use_ssl": ELASTICSEARCH_USE_SSL,
        "api_key": ELASTICSEARCH_API_KEY,
        "ca_certs": ELASTICSEARCH_CA_CERTS,
    },
}
