import json
import logging
from typing import Generator, Iterable, Optional, Union

import requests
from requests import Session
from requests.auth import HTTPBasicAuth

from django.core.exceptions import ImproperlyConfigured
from django.utils.functional import SimpleLazyObject

from mentor.utils.requests import requests_retry_session

from .settings import (
    SENDER_ACCOUNT_SLUG,
    SENDER_AUTH_KEY,
    SENDER_HOST,
    SENDER_SKIP_TEST,
)

log = logging.getLogger(__name__)


class Sender:
    base_path = "api/0"

    def __init__(
        self, host: str, account_slug: str, session: Optional[Session] = None, **kwargs
    ):
        self.host = host.rstrip("/")
        self.account_slug = account_slug
        self.session = session or Session()

        self.verify_host = kwargs.pop("verify_host", True)
        self.timeout = kwargs.pop("timeout", 10)
        self.retries = kwargs.pop("retries", 3)

        if self.retries:
            backoff_factor = kwargs.get("backoff_factor", 0.3)
            status_forcelist = kwargs.get("status_forcelist", (500, 502, 504))

            self.session = requests_retry_session(
                self.retries,
                backoff_factor,
                status_forcelist,
                self.session,
            )

    def request(self, method: str, url: str, data: dict = None, **kwargs) -> dict:
        url = f"https://{self.host}/{self.base_path}/{self.account_slug}/{url}"
        headers = {"Content-type": "application/json"}
        headers.update(**kwargs.pop("headers", {}))

        if SENDER_SKIP_TEST:
            log.debug(f"skipped sender request: {method} {url}\n{data} {headers}")
            return {}

        response = self.session.request(
            method,
            url,
            json=data,
            headers=headers,
            verify=self.verify_host,
            timeout=self.timeout,
            **kwargs,
        )
        response.raise_for_status()

        return response.json()

    def bulk_send_transactional_email(
        self,
        campaign_slug: str,
        data: Iterable[dict],
        **kwargs,
    ) -> Generator:
        for params in data:
            params.update(**kwargs)
            yield self.send_transactional_email(campaign_slug, **params)

    def send_transactional_email(
        self,
        campaign_slug: str,
        recipient: Union[str, dict],
        arguments: Optional[dict] = None,
        **kwargs,
    ):
        url = f"transactional/{campaign_slug}/send"

        data = {
            "async": True,
        }

        if arguments:
            data["args"] = json.dumps(arguments, ensure_ascii=False)

        if isinstance(recipient, dict):
            data.update(
                {
                    "to": recipient.get("to", []),
                    "cc": recipient.get("cc", []),
                    "bcc": recipient.get("bcc", []),
                }
            )

        elif isinstance(recipient, str):
            data["to_email"] = recipient

        data.update(**kwargs)

        return self.request("post", url, data)

    def get_message_status(self, campaign_slug: str, message_id: str):
        url = f"transactional/{campaign_slug}/status"
        params = {
            "message_id": message_id,
        }
        return self.request("get", url, params=params)


def get_client() -> Sender:
    if SENDER_HOST is None:
        raise ImproperlyConfigured(
            "Please set the SENDER_HOST in settings.py",
        )

    if SENDER_ACCOUNT_SLUG is None:
        raise ImproperlyConfigured(
            "Please set the SENDER_ACCOUNT_SLUG in settings.py",
        )

    if SENDER_AUTH_KEY is None:
        raise ImproperlyConfigured(
            "Please set the SENDER_AUTH_KEY in settings.py",
        )

    session = requests.session()
    session.auth = HTTPBasicAuth(  # nosec
        username=SENDER_AUTH_KEY,
        password="",
    )

    return Sender(
        host=SENDER_HOST,
        account_slug=SENDER_ACCOUNT_SLUG,
        session=session,
    )


sender: Union[SimpleLazyObject, Sender] = SimpleLazyObject(lambda: get_client())
