from ordered_model.admin import OrderedModelAdmin
from simple_history.admin import SimpleHistoryAdmin

from django.contrib import admin

from .models import (
    ExperienceType,
    MenteeFeedback,
    Mentor,
    MentorExperience,
    MentorFeedback,
    Mentorship,
    Position,
    Skill,
)


class MentorUserMixin:
    def display_mentor_user(self, obj):
        return obj.mentor.user

    display_mentor_user.short_description = "Ментор"
    display_mentor_user.admin_order_field = "mentor__user"


@admin.register(Skill)
class SkillAdmin(admin.ModelAdmin):
    list_display = ("name", "canonical_name", "created", "modified")
    search_fields = ("name", "canonical_name")


@admin.register(ExperienceType)
class ExperienceTypeAdmin(OrderedModelAdmin):
    list_display = ("name", "slug", "created", "modified", "move_up_down_links")
    search_fields = ("name", "slug")
    ordering = ("order",)

    readonly_fields = ("slug",)

    def has_add_permission(self, request) -> bool:
        return False

    def has_delete_permission(self, request, obj=None) -> bool:
        return False


class MentorExperienceInline(admin.StackedInline):
    model = MentorExperience
    fields = ("experience_type", "begin", "end", "name", "description")
    extra = 0


@admin.register(Mentor)
class MentorAdmin(admin.ModelAdmin):
    list_display = (
        "user",
        "is_published",
        "is_ready_for_mentorships",
        "mentees_count",
        "feedback_count",
        "created",
        "modified",
    )
    list_select_related = ("user",)
    list_filter = ("is_published", "is_ready_for_mentorships")
    search_fields = (
        "user__username",
        "user__yauid",
    )
    raw_id_fields = ("user",)
    readonly_fields = ("feedback_count", "mentees_count")
    inlines = [MentorExperienceInline]


@admin.register(Mentorship)
class MentorshipAdmin(MentorUserMixin, SimpleHistoryAdmin):
    list_display = ("display_mentor_user", "mentee", "status", "created", "modified")
    list_select_related = ("mentor__user", "mentee")
    list_filter = ("status",)
    search_fields = (
        "mentor__user__username",
        "mentor__user__yauid",
        "mentee__username",
        "mentee__yauid",
    )
    raw_id_fields = ("mentor", "mentee", "status_by")
    history_list_display = ["status"]


@admin.register(MentorFeedback)
class MentorFeedbackAdmin(MentorUserMixin, admin.ModelAdmin):
    list_display = ("display_mentor_user", "display_mentee", "created", "modified")
    list_select_related = ("mentor__user", "mentorship__mentee")
    search_fields = (
        "mentor__user__username",
        "mentor__user__yauid",
        "mentorship__mentee__username",
        "mentorship__mentee__yauid",
    )
    raw_id_fields = ("mentor", "mentorship")

    def display_mentee(self, obj: MentorFeedback):
        return obj.mentorship.mentee

    display_mentee.short_description = "Менти"
    display_mentee.admin_order_field = "mentorship__mentee"


@admin.register(MenteeFeedback)
class MenteeFeedbackAdmin(admin.ModelAdmin):
    list_display = (
        "mentee",
        "display_mentor_user",
        "created",
        "modified",
    )
    list_select_related = ("mentee", "mentorship__mentor__user")
    search_fields = (
        "mentee__username",
        "mentee__yauid",
        "mentorship__mentor__user__username",
        "mentorship__mentor__user__yauid",
    )
    raw_id_fields = ("mentee", "mentorship")

    def display_mentor_user(self, obj):
        return obj.mentorship.mentor.user

    display_mentor_user.admin_order_field = "mentorship__mentor__user"


@admin.register(Position)
class PositionAdmin(admin.ModelAdmin):
    list_display = ("name", "created", "modified")
