from ninja import Router
from ninja.errors import HttpError

from django.shortcuts import get_object_or_404
from django.utils.translation import gettext_lazy as _

from ..models import Mentor, MentorExperience
from ..schemas import ExperienceIn, ExperienceOut
from ..services import TransitionCheckResult

router = Router()


UPDATE_STATUS_ERROR_MESSAGES = {
    TransitionCheckResult.NOT_ALLOWED: (
        403,
        _("У Вас нет прав для изменения статуса {old_status}"),
    ),
    TransitionCheckResult.UNACCEPTABLE: (
        400,
        _("Переход из статуса {old_status} в {new_status} недопустим"),
    ),
}


@router.post(
    "/",
    response=ExperienceOut,
    tags=["Experiences"],
    summary=_("Создать опыт ментора"),
)
def create_experience(request, payload: ExperienceIn):
    mentor = get_object_or_404(Mentor, user=request.user)

    return MentorExperience.objects.create(**payload.dict(), mentor=mentor)


@router.patch(
    "/{experience_id}/",
    response=ExperienceOut,
    tags=["Experiences"],
    summary=_("Обновить опыт ментора"),
)
def update_experience(request, experience_id: int, payload: ExperienceIn):
    experience = get_object_or_404(MentorExperience, pk=experience_id)

    if experience.mentor.user != request.user:
        raise HttpError(
            403, _("У Вас нет прав для редактирования данного опыта ментора")
        )

    for key, value in payload.dict().items():
        setattr(experience, key, value)

    experience.save()

    return experience


@router.delete(
    "/{experience_id}/",
    response={204: None},
    tags=["Experiences"],
    summary=_("Удалить опыт ментора"),
)
def delete_experience(request, experience_id: int):
    experience = get_object_or_404(MentorExperience, pk=experience_id)

    if experience.mentor.user != request.user:
        raise HttpError(403, _("У Вас нет прав для удаленния данного опыта ментора"))

    experience.delete()

    return 204, None
