from ninja import Router
from ninja.errors import HttpError

from django.shortcuts import get_object_or_404
from django.utils.translation import gettext_lazy as _

from ..models import MenteeFeedback, Mentorship
from ..schemas import MenteeFeedbackIn, MenteeFeedbackOut

router = Router()


@router.post(
    "/",
    response=MenteeFeedbackOut,
    summary=_("Создать отзыв на менти"),
)
def create_mentee_feedback(request, payload: MenteeFeedbackIn):
    mentorship = get_object_or_404(
        Mentorship.objects, pk=payload.mentorship_id, mentor__user_id=request.user.id
    )

    if mentorship.status != Mentorship.Status.COMPLETED:
        raise HttpError(400, _("Нельзя создать отзыв, если менторство не завершено"))

    if mentorship.menteefeedback.exists():
        raise HttpError(400, _("Вы уже оставляли отзыв на данное менторство"))

    return MenteeFeedback.objects.create(
        **payload.dict(), mentee_id=mentorship.mentee_id
    )


@router.put(
    "/{feedback_id}/publish/",
    response=MenteeFeedbackOut,
    summary=_("Опубликовать отзыв на менти"),
)
def publish_mentee_feedback(request, feedback_id: int):
    mentee_feedback = get_object_or_404(
        MenteeFeedback, pk=feedback_id, mentee_id=request.user.id
    )

    if not mentee_feedback.can_publish:
        raise HttpError(403, _("Ментор запретил публиковать данный отзыв"))

    mentee_feedback.publish()

    return mentee_feedback


@router.put(
    "/{feedback_id}/unpublish/",
    response=MenteeFeedbackOut,
    summary=_("Снять отзыв на ментора с публикации"),
)
def unpublish_mentee_feedback(request, feedback_id: int):
    mentee_feedback = get_object_or_404(
        MenteeFeedback, pk=feedback_id, mentee_id=request.user.id
    )

    mentee_feedback.unpublish()

    return mentee_feedback
