from ninja import Router
from ninja.errors import HttpError

from django.db import transaction
from django.utils.translation import gettext_lazy as _

from ..models import Mentor
from ..schemas import MentorIn, MentorOut
from ..utils import create_skills

router = Router()


@router.post(
    "/",
    response=MentorOut,
    tags=["Mentor"],
    summary=_("Создать профиль ментора"),
    description=_("Создаёт ментора для текущего пользователя"),
)
@transaction.atomic
def create_mentor(request, payload: MentorIn):
    mentor, created = Mentor.objects.get_or_create(
        user=request.user,
        defaults=payload.dict(exclude={"skills"}),
    )

    if not created:
        raise HttpError(400, _("Профиль ментора уже существует"))

    skills = create_skills(payload.skills)
    mentor.skills.add(*skills)

    return mentor


@router.patch(
    "/",
    response=MentorOut,
    tags=["Mentor"],
    summary=_("Обновить профиль ментора"),
    description=_("Обновляет ментора для текущего пользователя"),
)
@transaction.atomic
def update_mentor(request, payload: MentorIn):
    mentor = Mentor.objects.select_for_update(nowait=True).get(user=request.user)

    if not mentor:
        raise HttpError(404, _("У Вас нет профиля ментора"))

    new_skills = create_skills(payload.skills)

    for key, value in payload.dict(exclude={"skills"}).items():
        setattr(mentor, key, value)

    mentor.save()

    mentor.skills.set(new_skills)

    return mentor
