from ninja import Router
from ninja.errors import HttpError

from django.shortcuts import get_object_or_404
from django.utils.translation import gettext_lazy as _

from ..models import MentorFeedback, Mentorship
from ..schemas import MentorFeedbackIn, MentorFeedbackOut

router = Router()


@router.post(
    "/",
    response=MentorFeedbackOut,
    summary=_("Создать отзыв на ментора"),
)
def create_mentor_feedback(request, payload: MentorFeedbackIn):
    mentorship = get_object_or_404(
        Mentorship, pk=payload.mentorship_id, mentee_id=request.user.id
    )

    if mentorship.status != Mentorship.Status.COMPLETED:
        raise HttpError(400, _("Нельзя создать отзыв, если менторство не завершено"))

    if mentorship.mentorfeedback.exists():
        raise HttpError(400, _("Вы уже оставляли отзыв на данное менторство"))

    return MentorFeedback.objects.create(
        **payload.dict(), mentor_id=mentorship.mentor_id
    )


@router.put(
    "/{feedback_id}/publish/",
    response=MentorFeedbackOut,
    summary=_("Опубликовать отзыв на ментора"),
)
def publish_mentor_feedback(request, feedback_id: int):
    mentor_feedback = get_object_or_404(
        MentorFeedback, pk=feedback_id, mentor__user=request.user
    )

    if not mentor_feedback.can_publish:
        raise HttpError(403, _("Менти запретил публиковать данный отзыв"))

    mentor_feedback.publish()

    return mentor_feedback


@router.put(
    "/{feedback_id}/unpublish/",
    response=MentorFeedbackOut,
    summary=_("Снять отзыв на ментора с публикации"),
)
def unpublish_mentor_feedback(request, feedback_id: int):
    mentor_feedback = get_object_or_404(
        MentorFeedback, pk=feedback_id, mentor__user=request.user
    )

    mentor_feedback.unpublish()

    return mentor_feedback
