from typing import List

from ninja import Router

from django.db.models import Q, Value
from django.db.models.functions import Concat
from django.shortcuts import get_object_or_404
from django.utils.translation import gettext_lazy as _

from mentor.core.schemas import PaginatedResponseSchema
from mentor.utils.pagination import paginated_response

from ..models import Mentor, MentorExperience, MentorFeedback
from ..schemas import (
    MentorDetailMentorOut,
    MentorListMentorOut,
    MentorsExperienceOut,
    MentorsMentorFeedbackOut,
)
from ..services import TransitionCheckResult, has_active_mentorship

router = Router()

UPDATE_STATUS_ERROR_MESSAGES = {
    TransitionCheckResult.NOT_ALLOWED: (
        403,
        _("У Вас нет прав для изменения статуса {old_status}"),
    ),
    TransitionCheckResult.UNACCEPTABLE: (
        400,
        _("Переход из статуса {old_status} в {new_status} недопустим"),
    ),
}


@router.get(
    "/",
    response=PaginatedResponseSchema[MentorListMentorOut],
    tags=["Mentors"],
    summary=_("Список менторов"),
)
def list_mentors(request, search: str = "", page: int = 1, limit: int = 10):
    mentors = (
        Mentor.objects.annotate(
            full_name=Concat("user__first_name", Value(" "), "user__last_name")
        )
        .filter(Q(full_name__icontains=search) | Q(description__icontains=search))
        .select_related(
            "user",
            "user__staff_profile",
            "user__staff_profile__city",
            "user__staff_profile__office",
        )
        .prefetch_related("user__staff_profile__groups")
        .order_by("pk")
    )

    return paginated_response(mentors, page, limit)


@router.get(
    "/{username}/",
    response=MentorDetailMentorOut,
    tags=["Mentors"],
    summary=_("Информация о менторе"),
)
def retrieve_mentor(request, username: str):
    mentor = get_object_or_404(
        Mentor.objects.select_related(
            "user",
            "user__staff_profile",
            "user__staff_profile__office",
            "user__staff_profile__city",
        ).prefetch_related("skills", "user__staff_profile__groups"),
        user__username=username,
    )

    mentor.has_active_mentorship = has_active_mentorship(
        mentee_id=request.user.pk, mentor_id=mentor.pk
    )

    return mentor


@router.get(
    "/{username}/feedback/",
    response=List[MentorsMentorFeedbackOut],
    tags=["Mentors"],
    summary=_("Список отзывов о менторе"),
)
def list_mentor_feedback(request, username: str):
    return (
        MentorFeedback.objects.visible()
        .filter(mentor__user__username=username)
        .select_related(
            "mentorship", "mentorship__mentee", "mentorship__mentee__staff_profile"
        )
    )


@router.get(
    "/{username}/experiences/",
    response=List[MentorsExperienceOut],
    tags=["Mentors"],
    summary=_("Опыт ментора"),
)
def list_experiences(request, username: str):
    mentor = get_object_or_404(Mentor, user__username=username)

    return MentorExperience.objects.filter(mentor=mentor).order_by(
        "experience_type__order"
    )
