from ninja import Router
from ninja.errors import HttpError

from django.db import transaction
from django.db.models import Q
from django.shortcuts import get_object_or_404
from django.utils.translation import gettext_lazy as _

from ..models import Mentor, Mentorship
from ..schemas import (
    MentorshipIn,
    MentorshipOut,
    MentorshipStatusIn,
    MentorshipStatusOut,
)
from ..services import TransitionCheckResult, check_transition, has_active_mentorship

router = Router()


UPDATE_STATUS_ERROR_MESSAGES = {
    TransitionCheckResult.NOT_ALLOWED: (
        403,
        _("У Вас нет прав для изменения статуса {old_status}"),
    ),
    TransitionCheckResult.UNACCEPTABLE: (
        400,
        _("Переход из статуса {old_status} в {new_status} недопустим"),
    ),
}


@router.post(
    "/",
    response=MentorshipOut,
    tags=["Mentorships"],
    summary=_("Создать менторство"),
)
def create_mentorship(request, payload: MentorshipIn):
    mentor = get_object_or_404(Mentor, pk=payload.mentor_id)

    if not mentor.is_ready_for_mentorships:
        raise HttpError(400, _("Ментор в данный момент не принимает запросы"))

    if has_active_mentorship(mentee_id=request.user.pk, mentor_id=payload.mentor_id):
        raise HttpError(
            400, _("У Вас уже есть незавершенное менторство с данным ментором")
        )

    return Mentorship.objects.create(
        **payload.dict(), mentee=request.user, status_by=request.user
    )


@router.patch(
    "/{mentorship_id}/status/",
    response=MentorshipStatusOut,
    tags=["Mentorships"],
    summary=_("Обновить статус менторства"),
)
@transaction.atomic
def update_mentorship_status(request, mentorship_id: int, payload: MentorshipStatusIn):
    queryset = Mentorship.objects.select_related("mentor").select_for_update(
        nowait=True
    )

    mentorship = get_object_or_404(
        queryset,
        Q(mentor__user=request.user) | Q(mentee=request.user),
        pk=mentorship_id,
    )

    check_result = check_transition(request, mentorship, payload.status)

    status_code, msg = UPDATE_STATUS_ERROR_MESSAGES.get(check_result, (None, None))

    if status_code:
        raise HttpError(
            status_code,
            msg.format(old_status=mentorship.status, new_status=payload.status),
        )

    for key, value in payload.dict().items():
        setattr(mentorship, key, value)

    mentorship.status_by = request.user
    mentorship.save()

    return mentorship


@router.delete(
    "/{mentorship_id}/",
    response={204: None},
    tags=["Mentorships"],
    summary=_("Удалить менторство"),
)
def remove_mentorship(request, mentorship_id: int):
    mentorship = get_object_or_404(
        Mentorship.objects.select_related("mentor", "mentor__user", "mentee"),
        Q(mentor__user=request.user) | Q(mentee=request.user),
        pk=mentorship_id,
    )

    if mentorship.status != Mentorship.Status.DECLINED:
        raise HttpError(
            400, _('Менторство можно удалить только из статуса "Отклонено"')
        )

    if mentorship.mentor.user == request.user:
        mentorship.remove_for_mentor()

    if mentorship.mentee == request.user:
        mentorship.remove_for_mentee()

    return 204, None
