from typing import List

from elasticsearch_dsl import Q
from ninja import Router

from django.utils.translation import gettext_lazy as _

from mentor.core.schemas import PaginatedResponseSchema
from mentor.mentorships.documents import (
    MentorExperienceDocument,
    PositionDocument,
    SkillDocument,
)
from mentor.mentorships.schemas import MentorListMentorOut
from mentor.mentorships.schemas.search import (
    MentorSearchIn,
    MentorSearchSuggestAllIn,
    MentorSearchSuggestAllOut,
    MentorSearchSuggestIn,
    MentorSearchSuggestOut,
)
from mentor.mentorships.services import (
    get_common_suggest_results,
    get_search_query,
    get_suggest_results,
)
from mentor.staff.documents import CityDocument, DepartmentDocument

router = Router()


@router.post(
    "/mentors/",
    response=PaginatedResponseSchema[MentorListMentorOut],
    summary=_("Поиск менторов"),
)
def search_mentors(request, search: MentorSearchIn, page: int = 1, limit: int = 10):
    s = get_search_query(search)

    # sorting
    s = s.sort("-rank", "_score")

    # pagination
    offset = (page - 1) * limit
    size = offset + limit
    s = s[offset:size]

    # convert to queryset
    queryset = (
        s.to_queryset()
        .select_related(
            "user",
            "user__staff_profile",
            "user__staff_profile__city",
            "user__staff_profile__office",
        )
        .prefetch_related("user__staff_profile__groups")
    )

    # count
    count = s.count()

    return {
        "count": count,
        "has_next": count > limit,
        "results": list(queryset),
    }


@router.post(
    "/suggest/skills/",
    response=List[MentorSearchSuggestOut],
    summary=_("Саджест по скиллам"),
)
def suggest_skills(request, search: MentorSearchSuggestIn):
    return get_suggest_results(
        SkillDocument,
        search,
        name="skills",
        field="skills.id",
    )


@router.post(
    "/suggest/departments/",
    response=List[MentorSearchSuggestOut],
    summary=_("Саджест по департаментам"),
)
def suggest_departments(request, search: MentorSearchSuggestIn):
    return get_suggest_results(
        DepartmentDocument,
        search,
        name="departments",
        field="departments.id",
        facet_size=40,
        size=20,
    )


@router.post(
    "/suggest/cities/",
    response=List[MentorSearchSuggestOut],
    summary=_("Саджест по городам"),
)
def suggest_cities(request, search: MentorSearchSuggestIn):
    return get_suggest_results(
        CityDocument,
        search,
        name="cities",
        field="city.id",
    )


@router.post(
    "/suggest/positions/",
    response=List[MentorSearchSuggestOut],
    summary=_("Саджест по должностям"),
)
def suggest_positions(request, search: MentorSearchSuggestIn):
    return get_suggest_results(
        PositionDocument,
        search,
        name="positions",
        field="position.id",
        facet_size=40,
        size=20,
    )


@router.post(
    "/suggest/employment/",
    response=List[MentorSearchSuggestOut],
    summary=_("Саджест по опыту работы ментора"),
)
def suggest_employment(request, search: MentorSearchSuggestIn):
    return get_suggest_results(
        MentorExperienceDocument,
        search,
        name="employments",
        field="employment.id",
        filters=[
            Q("term", experience_type="employment"),
        ],
    )


@router.post(
    "/suggest/education/",
    response=List[MentorSearchSuggestOut],
    summary=_("Саджест по образованию ментора"),
)
def suggest_education(request, search: MentorSearchSuggestIn):
    return get_suggest_results(
        MentorExperienceDocument,
        search,
        name="educations",
        field="education.id",
        filters=[
            Q("term", experience_type="education"),
        ],
    )


@router.post(
    "/suggest/all/",
    response=MentorSearchSuggestAllOut,
    summary=_("Саджест по основным полям"),
)
def suggest_all(request, search: MentorSearchSuggestAllIn):
    return get_common_suggest_results(search)
