import logging
from datetime import datetime

from django_elasticsearch_dsl import Document, fields
from django_elasticsearch_dsl.registries import registry
from elasticsearch_dsl import analyzer, tokenizer

from django.contrib.auth import get_user_model

from mentor.staff.models import StaffProfile

from .enums import ExperienceTypes
from .models import Mentor, MentorExperience, Mentorship, Position, Skill

logger = logging.getLogger(__name__)

User = get_user_model()

skill_analyzer = analyzer(
    "skill_analyzer",
    tokenizer=tokenizer("skill_tokenizer", "ngram", min_gram=2),
    filter=["lowercase"],
)


@registry.register_document
class SkillDocument(Document):
    name = fields.TextField(analyzer=skill_analyzer)

    class Index:
        name = "skills"

    class Django:
        model = Skill


@registry.register_document
class MentorDocument(Document):
    username = fields.TextField()
    first_name = fields.TextField()
    last_name = fields.TextField()
    position = fields.ObjectField(
        properties={
            "id": fields.KeywordField(),
            "name": fields.TextField(),
        }
    )
    city = fields.ObjectField(
        properties={
            "id": fields.KeywordField(),
            "name": fields.TextField(),
        }
    )
    departments = fields.ObjectField(
        multi=True,
        properties={
            "id": fields.KeywordField(),
            "level": fields.IntegerField(),
            "name": fields.TextField(),
        },
    )
    joined_at = fields.DateField()
    carrier_begin = fields.DateField()
    skills = fields.ObjectField(
        multi=True,
        properties={
            "id": fields.KeywordField(),
            "name": fields.TextField(),
        },
    )
    education = fields.ObjectField(
        multi=True,
        properties={
            "id": fields.KeywordField(),
            "name": fields.TextField(),
        },
    )
    employment = fields.ObjectField(
        multi=True,
        properties={
            "id": fields.KeywordField(),
            "name": fields.TextField(),
        },
    )
    is_chief = fields.BooleanField()
    has_feedback = fields.BooleanField()
    has_active_mentorships = fields.BooleanField()
    has_completed_mentorships = fields.BooleanField()
    rank = fields.IntegerField()

    def prepare_username(self, instance: Mentor):
        return instance.user.username

    def prepare_first_name(self, instance: Mentor):
        return instance.staff_profile.first_name

    def prepare_last_name(self, instance: Mentor):
        return instance.staff_profile.last_name

    def prepare_city(self, instance: Mentor):
        city = instance.staff_profile.city
        if city:
            return {
                "id": city.id,
                "name": city.name,
            }

    def prepare_position(self, instance: Mentor):
        position = instance.position
        if position:
            return {"id": position.id, "name": position.name}

    def prepare_departments(self, instance: Mentor):
        groups = []
        for group in instance.staff_profile.groups.all():
            for d in group.node.ancestors():
                groups.append(
                    {
                        "id": d.group.id,
                        "level": d.depth,
                        "name": d.group.name,
                    }
                )

        return groups

    def prepare_joined_at(self, instance: Mentor):
        joined_at = instance.staff_profile.joined_at
        if isinstance(joined_at, datetime):
            return joined_at.date()

        return None

    def prepare_skills(self, instance: Mentor):
        return [
            {
                "id": s.id,
                "name": s.name,
            }
            for s in instance.skills.all()
        ]

    def _get_experience_by_type(self, instance: Mentor, experience_type):
        return [
            {
                "id": e.id,
                "name": e.name,
            }
            for e in instance.experiences.filter(experience_type__slug=experience_type)
        ]

    def prepare_education(self, instance: Mentor):
        return self._get_experience_by_type(instance, ExperienceTypes.EDUCATION)

    def prepare_employment(self, instance: Mentor):
        return self._get_experience_by_type(instance, ExperienceTypes.EMPLOYMENT)

    def prepare_is_chief(self, instance: Mentor):
        return instance.staff_profile.is_head

    def prepare_has_feedback(self, instance: Mentor):
        return instance.feedback_count > 0

    def prepare_has_active_mentorships(self, instance: Mentor):
        return instance.mentees_count > 0

    def prepare_has_completed_mentorships(self, instance: Mentor):
        return bool(
            instance.mentorships.filter(status=Mentorship.Status.COMPLETED).exists()
        )

    def prepare_rank(self, instance: Mentor):
        """
        Возвращает рейтинг ментора

        Пока учитываем только заполнение профиля
        """
        rank = 100  # дефолтный рейтинг

        # описание
        if instance.description:
            rank += 10

        # поле "чем могу помочь"
        if len(list(instance.assistance)):
            rank += 10

        # указано образование
        education_exists = instance.experiences.filter(
            experience_type__slug=ExperienceTypes.EDUCATION
        ).exists()
        if education_exists:
            rank += 10

        # указан опыт работы
        employment_exists = instance.experiences.filter(
            experience_type__slug=ExperienceTypes.EMPLOYMENT
        ).exists()
        if employment_exists:
            rank += 10

        # указаны проекты
        projects_exists = instance.experiences.filter(
            experience_type__slug=ExperienceTypes.PROJECTS
        ).exists()
        if projects_exists:
            rank += 10

        # заполнены навыки
        if instance.skills.exists():
            rank += 10

        return rank

    class Index:
        name = "mentors"

    class Django:
        model = Mentor
        related_models = [MentorExperience, Skill, StaffProfile, User]

    def get_queryset(self):
        queryset = self.django.model.objects.all()

        return queryset.select_related(
            "user",
            "position",
            "user__staff_profile",
            "user__staff_profile__city",
        ).prefetch_related(
            "skills",
        )

    def get_instances_from_related(self, related_instance):
        if isinstance(related_instance, MentorExperience):  # type: MentorExperience
            return related_instance.mentor

        if isinstance(related_instance, Skill):  # type: Skill
            return related_instance.mentors.all()

        if isinstance(related_instance, Position):  # type: Position
            return related_instance.mentors.all()

        if isinstance(related_instance, StaffProfile):  # type: StaffProfile
            mentor = getattr(related_instance.user, "mentor", None)
            return mentor and mentor

        if isinstance(related_instance, User):  # type: User
            mentor = getattr(related_instance, "mentor", None)
            return mentor and mentor

    def _prepare_action(self, object_instance, action):
        # если ментор не виден, то удаляем его из индекса
        if not object_instance.is_visible:
            action = "delete"

        return super()._prepare_action(object_instance, action)

    def update(self, *args, **kwargs):
        # игнорируем 404, при повторном удалении ментора из индекса
        kwargs["ignore_status"] = (404,)
        super().update(*args, **kwargs)


@registry.register_document
class MentorExperienceDocument(Document):
    name = fields.TextField()
    experience_type = fields.KeywordField()

    def prepare_experience_type(self, instance: MentorExperience):
        return instance.experience_type.slug

    class Index:
        name = "mentor_experiences"

    class Django:
        model = MentorExperience

    def get_queryset(self):
        qs = super().get_queryset()

        return qs.select_related(
            "experience_type",
        )


@registry.register_document
class PositionDocument(Document):
    name = fields.TextField()

    class Index:
        name = "mentor_positions"

    class Django:
        model = Position
