import logging
from typing import Iterable

from tqdm import tqdm

from django.contrib.auth import get_user_model
from django.core.management.base import BaseCommand
from django.db import transaction

from mentor.mentorships.models import Mentor
from mentor.mentorships.services import (
    update_mentor_feedback_count,
    update_mentor_mentees_count,
)

User = get_user_model()


class DryRunException(Exception):
    pass


class Command(BaseCommand):
    def update_mentees_count(self, mentor_ids: Iterable[int]) -> None:
        self.stdout.write("Updating mentees_count of mentors")

        for mentor_id in tqdm(mentor_ids):
            update_mentor_mentees_count(mentor_id)

    def update_feedback_count(self, mentor_ids: Iterable[int]):
        self.stdout.write("Updating feedback_count of mentors")

        for mentor_id in tqdm(mentor_ids):
            update_mentor_feedback_count(mentor_id)

    def update(self, options):
        mentor_ids = Mentor.objects.values_list("id", flat=True)

        if not options.get("skip_mentees_count_update", False):
            self.update_mentees_count(mentor_ids)

        if not options.get("skip_feedback_count_update", False):
            self.update_feedback_count(mentor_ids)

    def add_arguments(self, parser):
        parser.add_argument(
            "--skip-mentees-count-update",
            action="store_true",
            dest="skip_mentees_count_update",
            help="Disable updating mentees_count field",
        )
        parser.add_argument(
            "--skip-feedback-count-update",
            action="store_true",
            dest="skip_feedback_count_update",
            help="Disable updating feedback_count field",
        )
        parser.add_argument(
            "--dry-run",
            action="store_true",
            dest="dry_run",
            help="Run without commit to db",
        )

    def handle(self, *args, **options):
        # log only if level is more severe than or equal to WARNING
        logging.disable(logging.INFO)

        try:
            with transaction.atomic():
                self.update(options)

            if options.get("dry_run", False):
                raise DryRunException

        except DryRunException:
            self.stdout.write("Rollback")

        else:
            self.stdout.write("Commit")
